<?php

declare(strict_types=1);

namespace ContentReady\Admin\Dashboard;

use ContentReady\Modules\Registry;

final class Completion
{
	/**
	 * 判断某篇文章是否可计入“已生成 GEO”。
	 *
	 * 设计目标：完成度判定稳定、可预测，避免随模块增多而频繁调整。
	 * 判定逻辑：
	 * - 必须存在顶层 generated_at
	 * - 优先使用 last_updated 作为“已生成”标记（稳定/本地生成/always）
	 * - 若 last_updated 不在当前页面类型或被禁用，则回退到 summary（若启用）
	 * - 若两者都不可用，则仅以 generated_at 作为最弱判定
	 */
	public static function is_post_geo_complete(array $meta, array $options, array $modules_config): bool
	{
		$generated_at = (int)($meta['generated_at'] ?? 0);
		if ($generated_at <= 0) {
			return false;
		}

		$checkpoint = self::get_completion_checkpoint_module_id($meta, $options, $modules_config);
		if ($checkpoint === '') {
			return true;
		}

		$data = $meta['modules'][$checkpoint]['data'] ?? [];
		return self::is_module_data_present($checkpoint, is_array($data) ? $data : []);
	}

	/**
	 * 获取某篇文章当前页面类型下启用的内容模块（不含 slug_suggestions）。
	 *
	 * @return string[]
	 */
	public static function get_enabled_content_modules_for_post(array $meta, array $options, array $modules_config): array
	{
		$page_type = is_string($meta['page_type'] ?? null) ? (string)$meta['page_type'] : 'article';
		$page_types = is_array($options['page_types'] ?? null) ? $options['page_types'] : [];

		$pt_modules = [];
		foreach ($page_types as $row) {
			if (!is_array($row)) {
				continue;
			}
			if (($row['id'] ?? null) === $page_type && is_array($row['modules'] ?? null)) {
				$pt_modules = array_values($row['modules']);
				break;
			}
		}

		if ($pt_modules === []) {
			$pt_modules = array_values(array_filter(Registry::all_ids(), [Registry::class, 'is_content_module']));
		}

		$out = [];
		foreach ($pt_modules as $module_id) {
			$module_id = sanitize_key((string)$module_id);
			if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
				continue;
			}
			if ($module_id === 'slug_suggestions') {
				continue;
			}
			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}
			$out[] = $module_id;
		}

		return $out;
	}

	/**
	 * 获取“完成度标记模块”ID（返回空字符串表示仅用 generated_at 判定）。
	 */
	private static function get_completion_checkpoint_module_id(array $meta, array $options, array $modules_config): string
	{
		$enabled = array_flip(self::get_enabled_content_modules_for_post($meta, $options, $modules_config));
		if (isset($enabled['last_updated'])) {
			return 'last_updated';
		}
		if (isset($enabled['summary'])) {
			return 'summary';
		}
		return '';
	}

	/**
	 * 判断模块 data 是否“有意义地存在”。
	 */
	private static function is_module_data_present(string $module_id, array $data): bool
	{
		if ($module_id === 'last_updated') {
			return trim((string)($data['date'] ?? '')) !== '';
		}
		if ($module_id === 'summary' || $module_id === 'verdict') {
			return trim((string)($data['text'] ?? '')) !== '';
		}
		return $data !== [];
	}
}

