<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\AI\Client;

final class AiProbe
{
	private const NONCE_ACTION = 'cr_probe_ai';
	private const EXPECTED_ECHO = '欢迎使用 Content Ready｜内容就绪 插件';
	private const TIMEOUT_MIN_SEC = 20;
	private const TIMEOUT_MAX_SEC = 300;

	public static function register(): void
	{
		add_action('wp_ajax_cr_probe_ai', [self::class, 'handle_ajax']);
	}

	public static function handle_ajax(): void
	{
		if (!current_user_can('manage_options')) {
			wp_send_json_error(['code' => 'permission_denied', 'message' => '权限不足：仅管理员可执行 AI 测通。'], 403);
		}

		check_ajax_referer(self::NONCE_ACTION);

		$base_url = isset($_POST['ai_base_url']) ? esc_url_raw(trim((string)wp_unslash($_POST['ai_base_url']))) : '';
		$api_key = isset($_POST['ai_api_key']) ? sanitize_text_field(trim((string)wp_unslash($_POST['ai_api_key']))) : '';
		$model = isset($_POST['ai_model']) ? sanitize_text_field(trim((string)wp_unslash($_POST['ai_model']))) : '';
		$timeout_sec = isset($_POST['ai_request_timeout_sec']) ? (int)wp_unslash((string)$_POST['ai_request_timeout_sec']) : 90;
		$timeout_sec = max(self::TIMEOUT_MIN_SEC, min(self::TIMEOUT_MAX_SEC, $timeout_sec));

		if ($base_url === '' || $api_key === '' || $model === '') {
			self::persist_probe_result(false, '请先填写 Base URL、AI Key、模型，再点击测通。');
			wp_send_json_error([
				'code' => 'config_missing',
				'message' => '请先填写 Base URL、AI Key、模型，再点击测通。',
			], 400);
		}

		$client = new Client($base_url, $api_key, $model);
		if (!$client->is_configured()) {
			self::persist_probe_result(false, 'AI 配置不完整：请检查 Base URL 与模型。');
			wp_send_json_error([
				'code' => 'config_invalid',
				'message' => 'AI 配置不完整：请检查 Base URL 与模型。',
			], 400);
		}

		$system = '你是连通性测试助手。你必须且只能返回 JSON 对象，不要输出任何解释。';
		$user = '请将【' . self::EXPECTED_ECHO . '】括号里的原文完整放入 echo 字段，并只返回 JSON：{"echo":"' . self::EXPECTED_ECHO . '"}';

		$response = $client->chat_json($system, $user, $timeout_sec);
		if (empty($response['ok'])) {
			$raw_error = is_string($response['error'] ?? null) ? (string)$response['error'] : 'ai_request_failed';
			$message = 'AI 调用失败：' . $raw_error;
			self::persist_probe_result(false, $message);
			$status = isset($response['status']) ? (int)$response['status'] : 502;
			if ($status < 400 || $status > 599) {
				$status = 502;
			}
			wp_send_json_error([
				'code' => 'request_failed',
				'message' => $message,
				'raw_error' => $raw_error,
			], $status);
		}

		$payload = is_array($response['json'] ?? null) ? $response['json'] : [];
		$echo_text = isset($payload['echo']) ? trim((string)$payload['echo']) : '';
		if ($echo_text !== self::EXPECTED_ECHO) {
			$message = 'AI 已响应，但回显内容不符合预期。请检查模型是否支持稳定遵循指令。';
			self::persist_probe_result(false, $message);
			wp_send_json_error([
				'code' => 'echo_mismatch',
				'message' => $message,
				'expected' => self::EXPECTED_ECHO,
				'actual' => $echo_text,
			], 200);
		}

		$message = '测通成功：AI 可用，且已正确回显固定内容。';
		self::persist_probe_result(true, $message);
		wp_send_json_success([
			'message' => $message,
			'echo' => $echo_text,
			'attempts' => 1,
		]);
	}

	private static function persist_probe_result(bool $ok, string $message): void
	{
		$options = wp_parse_args(Settings::get_raw_options(), Settings::default_options());
		$ai = is_array($options['ai'] ?? null) ? $options['ai'] : [];
		$ai['probe_last_at'] = time();
		$ai['probe_last_ok'] = $ok;
		$ai['probe_last_message'] = sanitize_text_field($message);
		$options['ai'] = $ai;
		Settings::update_options($options);
	}
}
