<?php

declare(strict_types=1);

namespace ContentReady\AI;

final class Client
{
	private string $base_url;
	private string $api_key;
	private string $model;

	public function __construct(string $base_url, string $api_key, string $model)
	{
		$this->base_url = rtrim($base_url, '/');
		$this->api_key = $api_key;
		$this->model = $model;
	}

	public function is_configured(): bool
	{
		return $this->base_url !== '' && $this->model !== '';
	}

	public function chat_json(string $system, string $user, int $timeout = 30): array
	{
		$timeout = max(5, min(300, $timeout));
		$result = $this->chat_json_once($system, $user, $timeout);
		$result['attempts'] = 1;
		return $result;
	}

	private function chat_json_once(string $system, string $user, int $timeout): array
	{
		$endpoint = $this->base_url;
		if (substr($endpoint, -3) === '/v1') {
			$endpoint .= '/chat/completions';
		} elseif (strpos($endpoint, '/v1/') !== false) {
			$endpoint = rtrim($endpoint, '/') . '/chat/completions';
		} else {
			$endpoint = rtrim($endpoint, '/') . '/v1/chat/completions';
		}

		$headers = [
			'Content-Type' => 'application/json',
		];
		if ($this->api_key !== '') {
			$headers['Authorization'] = 'Bearer ' . $this->api_key;
		}

		$body = [
			'model' => $this->model,
			'messages' => [
				['role' => 'system', 'content' => $system],
				['role' => 'user', 'content' => $user],
			],
			'response_format' => [
				'type' => 'json_object',
			],
		];

		$response = wp_remote_post($endpoint, [
			'timeout' => $timeout,
			'headers' => $headers,
			'body' => wp_json_encode($body, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
		]);

		if (is_wp_error($response)) {
			return [
				'ok' => false,
				'error' => $response->get_error_message(),
			];
		}

		$code = (int)wp_remote_retrieve_response_code($response);
		$raw = (string)wp_remote_retrieve_body($response);
		$decoded = $raw !== '' ? json_decode($raw, true) : null;

		if (!is_array($decoded)) {
			return [
				'ok' => false,
				'error' => 'invalid_json_response',
				'status' => $code,
				'raw' => $raw,
			];
		}

		if ($code < 200 || $code >= 300) {
			$message = 'http_' . $code;
			if (isset($decoded['error']['message']) && is_string($decoded['error']['message'])) {
				$message = $decoded['error']['message'];
			}
			return [
				'ok' => false,
				'error' => $message,
				'status' => $code,
			];
		}

		$content = $decoded['choices'][0]['message']['content'] ?? null;
		if (!is_string($content) || trim($content) === '') {
			return [
				'ok' => false,
				'error' => 'empty_model_content',
			];
		}

		$json = json_decode($content, true);
		if (!is_array($json)) {
			return [
				'ok' => false,
				'error' => 'model_output_not_json',
				'raw' => $content,
			];
		}

		return [
			'ok' => true,
			'json' => $json,
		];
	}
}
