(function () {
	function safeItems(checkResult) {
		return checkResult && Array.isArray(checkResult.items) ? checkResult.items : [];
	}

	function normalizeSummary(checkResult) {
		var s = (checkResult && checkResult.summary) || null;
		var fail = s && typeof s.fail === 'number' ? s.fail : null;
		var warn = s && typeof s.warn === 'number' ? s.warn : null;
		var pass = s && typeof s.pass === 'number' ? s.pass : null;

		if (fail === null || warn === null || pass === null) {
			fail = 0;
			warn = 0;
			pass = 0;
			safeItems(checkResult).forEach(function (it) {
				var level = (it && it.level) || '';
				if (level === 'fail') fail++;
				else if (level === 'warn') warn++;
				else if (level === 'pass') pass++;
			});
		}

		return { fail: fail || 0, warn: warn || 0, pass: pass || 0 };
	}

	function groupItems(checkResult) {
		var out = { fail: [], warn: [], pass: [] };
		safeItems(checkResult).forEach(function (it) {
			var level = (it && it.level) || 'warn';
			if (!out[level]) level = 'warn';
			out[level].push(it);
		});
		return out;
	}

	function renderSummary(el, checkResult) {
		if (!checkResult) return null;
		var s = normalizeSummary(checkResult);
		return el('p', null, '未达标：' + s.fail + ' / 警告：' + s.warn + ' / 通过：' + s.pass);
	}

	function renderItems(el, Button, items) {
		if (!Array.isArray(items) || !items.length) return null;
		return el(
			'ul',
			{ style: { paddingLeft: '18px' } },
			items.map(function (it, idx) {
				var level = it.level || '';
				var color = level === 'fail' ? '#b91c1c' : level === 'warn' ? '#b45309' : '#065f46';
				var moduleId = it && it.module_id;
				return el(
					'li',
					{ key: idx, style: { color: color, display: 'flex', gap: '8px', alignItems: 'center' } },
					el('span', null, it.message || it.code || 'unknown'),
					moduleId
						? el(
							Button,
							{
								isSecondary: true,
								isSmall: true,
								onClick: function () {
									if (window.CR_EDITOR_UTILS && window.CR_EDITOR_UTILS.scrollToModule) {
										window.CR_EDITOR_UTILS.scrollToModule(moduleId);
									}
								},
							},
							'跳转'
						)
						: null
				);
			})
		);
	}

	function renderGroup(el, Button, title, items, openByDefault) {
		if (!Array.isArray(items) || !items.length) return null;
		return el(
			'details',
			{ open: !!openByDefault, style: { marginTop: '8px' } },
			el('summary', { style: { cursor: 'pointer', fontWeight: 600 } }, title + '（' + items.length + '）'),
			renderItems(el, Button, items)
		);
	}

	window.CR_EDITOR_CHECKS = {
		runCheck: function (api, postId, setError, setLoading, setCheckResult) {
			setError('');
			setLoading(true);
			return api('/post/' + postId + '/check', { method: 'POST' })
				.then(function (res) {
					setCheckResult(res && res.result ? res.result : null);
				})
				.catch(function (e) {
					setError((e && e.message) || '检查失败');
				})
				.finally(function () {
					setLoading(false);
				});
		},
		renderPanel: function (el, PanelBody, Button, checkResult, onRun, loading) {
			var groups = groupItems(checkResult);
			return el(
				PanelBody,
				{ title: '发布前检查', initialOpen: false },
				el(Button, { isSecondary: true, onClick: onRun, disabled: loading }, '开始检查'),
				renderSummary(el, checkResult),
				renderGroup(el, Button, '未达标', groups.fail, true),
				renderGroup(el, Button, '警告', groups.warn, true),
				renderGroup(el, Button, '通过', groups.pass, false)
			);
		},
	};
})();

