(function () {
  function byId(id) {
    return document.getElementById(id);
  }

  function initSiteLogoPicker() {
    var input = byId('cr_site_logo_url');
    var pickBtn = byId('cr_site_logo_pick');
    var clearBtn = byId('cr_site_logo_clear');
    var preview = byId('cr_site_logo_preview');

    if (!input || !pickBtn || !preview) return;

    function renderPreview(url) {
      preview.innerHTML = '';
      if (!url) return;

      var img = document.createElement('img');
      img.src = url;
      img.alt = '';
      img.style.maxWidth = '120px';
      img.style.height = 'auto';
      img.style.border = '1px solid #e5e7eb';
      img.style.borderRadius = '8px';
      img.style.padding = '6px';
      img.style.background = '#fff';
      preview.appendChild(img);
    }

    renderPreview(input.value);

    pickBtn.addEventListener('click', function (e) {
      e.preventDefault();

      if (!window.wp || !window.wp.media) {
        // wp.media 未加载时不阻塞页面使用
        return;
      }

      var frame = window.wp.media({
        title: '选择网站 Logo',
        button: { text: '使用该图片' },
        library: { type: 'image' },
        multiple: false,
      });

      frame.on('select', function () {
        var selection = frame.state().get('selection');
        if (!selection) return;

        var attachment = selection.first();
        if (!attachment) return;

        var data = attachment.toJSON();
        if (!data || !data.url) return;

        input.value = data.url;
        renderPreview(data.url);
      });

      frame.open();
    });

    if (clearBtn) {
      clearBtn.addEventListener('click', function (e) {
        e.preventDefault();
        input.value = '';
        renderPreview('');
      });
    }
  }

  function initModulesSearch() {
    var input = byId('cr-modules-search');
    var wrap = byId('cr-modules-cards');
    if (!input || !wrap) return;

    var empty = byId('cr-modules-search-empty');
    var count = byId('cr-modules-search-count');
    var cards = Array.prototype.slice.call(wrap.querySelectorAll('.cr-module-card'));
    var groups = Array.prototype.slice.call(wrap.querySelectorAll('.cr-modules-group'));

    function norm(s) {
      return String(s || '').toLowerCase();
    }

    function cardHaystack(card) {
      return (
        norm(card.getAttribute('data-module-id')) +
        ' ' +
        norm(card.getAttribute('data-module-label')) +
        ' ' +
        norm(card.getAttribute('data-module-intro'))
      );
    }

    function update() {
      var q = norm(input.value).trim();
      var visible = 0;

      cards.forEach(function (card) {
        var show = !q || cardHaystack(card).indexOf(q) >= 0;
        card.style.display = show ? '' : 'none';
        if (show) visible++;
      });

      groups.forEach(function (group) {
        var groupCards = group.querySelectorAll('.cr-module-card');
        var any = false;
        for (var i = 0; i < groupCards.length; i++) {
          if (groupCards[i].style.display !== 'none') {
            any = true;
            break;
          }
        }
        group.style.display = any ? '' : 'none';
      });

      if (count) {
        count.textContent = q ? '匹配：' + visible + ' / ' + cards.length : '共 ' + cards.length + ' 个模块';
      }
      if (empty) {
        empty.style.display = visible ? 'none' : 'block';
      }
    }

    input.addEventListener('input', update);
    update();
  }

  function initDispatchModeUX() {
    var select = byId('cr-generation-dispatch-mode');
    var wrap = byId('cr-modules-cards');
    if (!select || !wrap) return;

    var enableAllBtn = byId('cr-adaptive-enable-all');
    var enableAllWrap = byId('cr-adaptive-enable-all-wrap');

    function isLockedCard(card) {
      return card && card.classList && card.classList.contains('cr-module-card--locked');
    }

    function setModulesDisabled(disabled) {
      var cards = Array.prototype.slice.call(wrap.querySelectorAll('.cr-module-card'));
      cards.forEach(function (card) {
        if (isLockedCard(card)) return;
        var controls = card.querySelectorAll('input, textarea, select');
        for (var i = 0; i < controls.length; i++) {
          controls[i].disabled = !!disabled;
        }
      });
    }

    function enableAdaptiveDefaults() {
      var boxes = wrap.querySelectorAll('input[type=checkbox][data-cr-adaptive-default="1"]');
      for (var i = 0; i < boxes.length; i++) {
        if (boxes[i].disabled) continue;
        boxes[i].checked = true;
      }
    }

    var lastMode = String(select.value || '');

    function update() {
      var mode = String(select.value || '');

      // 轻量精选：卡片只读（避免站长误以为这些开关会生效）
      var isLightweight = mode === 'lightweight';
      setModulesDisabled(isLightweight);

      // 自适应：切换后默认启用“候选池”模块
      if (mode === 'adaptive' && lastMode !== 'adaptive') {
        enableAdaptiveDefaults();
      }

      // 仅在自适应模式下显示“重新启用全部候选模块”按钮
      if (enableAllWrap) {
        enableAllWrap.style.display = mode === 'adaptive' ? '' : 'none';
      }

      lastMode = mode;
    }

    select.addEventListener('change', update);
    if (enableAllBtn) {
      enableAllBtn.addEventListener('click', function (e) {
        e.preventDefault();
        enableAdaptiveDefaults();
      });
    }

    // 初始化（不触发自适应默认启用，避免覆盖用户已关闭的模块）
    if (enableAllWrap) {
      enableAllWrap.style.display = String(select.value || '') === 'adaptive' ? '' : 'none';
    }
  }

  function initAiProbe() {
    var btn = byId('cr-ai-probe-btn');
    if (!btn || !window.ajaxurl) return;

    var msg = byId('cr-ai-probe-msg');
    var nonce = btn.getAttribute('data-cr-nonce');
    var baseUrlInput = byId('cr_ai_base_url');
    var apiKeyInput = byId('cr_ai_api_key');
    var modelInput = byId('cr_ai_model');
    var timeoutInput = byId('cr_ai_request_timeout_sec');
    if (!nonce || !baseUrlInput || !apiKeyInput || !modelInput) return;

    function setMsg(text, type) {
      if (!msg) return;
      msg.textContent = text || '';
      if (type === 'error') {
        msg.style.color = '#b91c1c';
      } else if (type === 'success') {
        msg.style.color = '#065f46';
      } else {
        msg.style.color = '';
      }
    }

    btn.addEventListener('click', function (e) {
      e.preventDefault();

      if (btn.getAttribute('data-cr-loading') === '1') return;

      var baseUrl = String(baseUrlInput.value || '').trim();
      var apiKey = String(apiKeyInput.value || '').trim();
      var model = String(modelInput.value || '').trim();
      var timeoutSec = timeoutInput ? String(timeoutInput.value || '').trim() : '90';
      if (!baseUrl || !apiKey || !model) {
        setMsg('请先填写 Base URL、AI Key、模型。', 'error');
        return;
      }

      btn.setAttribute('data-cr-loading', '1');
      var oldText = btn.textContent;
      btn.textContent = '测通中...';
      btn.setAttribute('aria-disabled', 'true');
      setMsg('正在发起真实调用，请稍候...', '');

      var body = new URLSearchParams();
      body.append('action', 'cr_probe_ai');
      body.append('_wpnonce', nonce);
      body.append('ai_base_url', baseUrl);
      body.append('ai_api_key', apiKey);
      body.append('ai_model', model);
      body.append('ai_request_timeout_sec', timeoutSec);

      fetch(window.ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
        body: body.toString(),
      })
        .then(function (r) {
          return r.json();
        })
        .then(function (json) {
          if (!json || !json.success) {
            throw new Error((json && json.data && json.data.message) || '测通失败');
          }
          var data = json.data || {};
          setMsg(String(data.message || '测通成功'), 'success');
        })
        .catch(function (err) {
          setMsg((err && err.message) || '测通失败，请稍后重试。', 'error');
        })
        .finally(function () {
          btn.textContent = oldText;
          btn.setAttribute('data-cr-loading', '0');
          btn.removeAttribute('aria-disabled');
        });
    });
  }

  function initJsonLdProbe() {
    var btn = byId('cr-jsonld-probe-btn');
    if (!btn || !window.ajaxurl) return;

    var msg = byId('cr-jsonld-probe-msg');
    var nonce = btn.getAttribute('data-cr-nonce');
    if (!nonce) return;

    function formatScopeState(row) {
      row = row || {};

      var ts = Number(row.checked_at || 0);
      var other = Number(row.other_scripts || 0);
      var mode = String(row.mode || '');
      var br = String(row.blocked_reason || '');
      var wo = !!row.would_output;

      var state = '未检测';
      if (ts > 0) {
        if (br === 'structured_disabled') {
          state = '已关闭';
        } else if (br === 'no_objects_enabled') {
          state = '无输出对象';
        } else if (!wo && br === 'auto_not_checked_yet') {
          state = '等待自检';
        } else if (!wo && br === 'auto_check_failed') {
          state = '自检失败（已暂停）';
        } else if (!wo && br === 'auto_conflict_detected') {
          state = other > 0 ? '已暂停输出（当前检测到 ' + other + ' 个其他来源的 JSON-LD）' : '已暂停输出（检测到其他来源的 JSON-LD）';
        } else if (wo) {
          if (mode === 'force') {
            state =
              other > 0
                ? '强制输出中（当前检测到 ' + other + ' 个其他来源的 JSON-LD）'
                : '强制输出中（未检测到其他来源的 JSON-LD）';
          } else {
            state = '正常输出（未检测到其他来源的 JSON-LD）';
          }
        } else {
          state = br ? '已暂停（' + br + '）' : '已暂停';
        }

        // 只显示时分秒：更直观看到刷新是否生效
        var d = new Date(ts * 1000);
        var hh = String(d.getHours()).padStart(2, '0');
        var mm = String(d.getMinutes()).padStart(2, '0');
        var ss = String(d.getSeconds()).padStart(2, '0');
        state += ' · ' + hh + ':' + mm + ':' + ss;
      }

      return state;
    }

    function updateScopeStates(scopes) {
      if (!scopes) return;
      Object.keys(scopes).forEach(function (sid) {
        var el = document.querySelector('[data-cr-jsonld-scope="' + sid + '"]');
        if (!el) return;
        el.textContent = formatScopeState(scopes[sid]);
      });
    }

    btn.addEventListener('click', function (e) {
      // JS 可用时走 AJAX；失败则回退到默认跳转（href）
      e.preventDefault();

      if (btn.getAttribute('data-cr-loading') === '1') return;
      btn.setAttribute('data-cr-loading', '1');

      var oldText = btn.textContent;
      btn.textContent = '探测中...';
      btn.setAttribute('aria-disabled', 'true');

      if (msg) {
        msg.textContent = '正在探测...';
      }

      var body = new URLSearchParams();
      body.append('action', 'cr_probe_jsonld');
      body.append('_wpnonce', nonce);

      fetch(window.ajaxurl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
        body: body.toString(),
      })
        .then(function (r) {
          return r.json();
        })
        .then(function (json) {
          if (!json || !json.success) {
            throw new Error((json && json.data && json.data.message) || '探测失败');
          }

          var status = (json.data && json.data.status) || {};
          var scopes = status.scopes || {};
          updateScopeStates(scopes);

          var result = (json.data && json.data.result) || {};
          var ok = Number(result.ok || 0);
          var skipped = Number(result.skipped || 0);
          var failed = Number(result.failed || 0);
          var total = Number(result.total || 0);

          if (msg) {
            msg.textContent =
              '已完成：成功 ' +
              ok +
              '，跳过 ' +
              skipped +
              '，失败 ' +
              failed +
              (total > 0 ? '（共 ' + total + '）' : '') +
              '。';
          }
        })
        .catch(function (err) {
          if (msg) {
            msg.textContent = (err && err.message) || '探测失败';
            msg.style.color = '#b91c1c';
          }
        })
        .finally(function () {
          btn.textContent = oldText;
          btn.setAttribute('data-cr-loading', '0');
          btn.removeAttribute('aria-disabled');
        });
    });
  }

  initSiteLogoPicker();
  initModulesSearch();
  initDispatchModeUX();
  initAiProbe();
  initJsonLdProbe();
})();

