<?php

declare(strict_types=1);

namespace ContentReady\Util;

final class Locale
{
	/**
	 * 将 WordPress locale（如 zh_CN / en_US）转换为 BCP47（如 zh-CN / en-US）。
	 */
	public static function wp_locale_to_bcp47(string $wp_locale): string
	{
		$wp_locale = trim($wp_locale);
		if ($wp_locale === '') {
			return '';
		}

		$tag = str_replace('_', '-', $wp_locale);
		return self::normalize_bcp47($tag);
	}

	/**
	 * 规范化并校验 BCP47 语言标签（v1 仅支持 lang 或 lang-REGION）。
	 *
	 * 示例：
	 * - zh-cn -> zh-CN
	 * - en_US -> en-US
	 */
	public static function normalize_bcp47(string $tag): string
	{
		$tag = trim($tag);
		if ($tag === '') {
			return '';
		}

		$tag = str_replace('_', '-', $tag);
		$parts = array_values(array_filter(explode('-', $tag), static fn($p) => $p !== ''));
		if (count($parts) < 1 || count($parts) > 2) {
			return '';
		}

		$lang = strtolower($parts[0]);
		if (!preg_match('/^[a-z]{2,3}$/', $lang)) {
			return '';
		}

		if (count($parts) === 1) {
			return $lang;
		}

		$region = strtoupper($parts[1]);
		if (!preg_match('/^[A-Z]{2}$/', $region)) {
			return '';
		}

		return $lang . '-' . $region;
	}
}

