<?php

declare(strict_types=1);

namespace ContentReady\Toc;

use ContentReady\Admin\Settings;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Sanitizer;

final class TocGenerator
{
	public static function register(): void
	{
		add_action('save_post', [self::class, 'on_save_post'], 35, 3);
	}

	public static function on_save_post(int $post_id, \WP_Post $post, bool $update): void
	{
		if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
			return;
		}
		if (!in_array($post->post_status, ['publish', 'future'], true)) {
			return;
		}
		if (!current_user_can('edit_post', $post_id)) {
			return;
		}

		$options = Settings::get_options();
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$toc_cfg = is_array($modules_cfg['toc'] ?? null) ? $modules_cfg['toc'] : [];
		if (empty($toc_cfg['enabled'])) {
			return;
		}

		$html = (string)$post->post_content;
		$meta = MetaStore::get($post_id);

		// 若 toc 曾被手动编辑，则不自动覆盖。
		$m = is_array($meta['modules']['toc'] ?? null) ? $meta['modules']['toc'] : [];
		$gen_at = (int)($m['generated_at'] ?? 0);
		$edit_at = (int)($m['edited_at'] ?? 0);
		if ($edit_at > 0 && $gen_at > 0 && $edit_at > $gen_at) {
			return;
		}

		$items = TocBuilder::build_items($html, $meta);
		$next = Sanitizer::sanitize_module_data('toc', ['items' => $items]);

		$prev = is_array($m['data'] ?? null) ? $m['data'] : [];
		$prev_json = wp_json_encode($prev);
		$next_json = wp_json_encode($next);
		if (is_string($prev_json) && is_string($next_json) && $prev_json === $next_json) {
			return;
		}

		MetaStore::touch_generated($post_id, 'toc', $next);
	}
}

