<?php

declare(strict_types=1);

namespace ContentReady\Rest;

use ContentReady\Admin\Settings;
use ContentReady\Checks\Checker;
use ContentReady\Frontend\StructuredData;
use ContentReady\Meta\MetaStore;
use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Registry;
use ContentReady\Modules\Sanitizer;
use ContentReady\Generator\V1Generator;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class PostController
{
	public static function register(): void
	{
		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/meta', [
			[
				'methods' => 'GET',
				'callback' => [self::class, 'get_meta'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/page-type', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'set_page_type'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'page_type' => ['type' => 'string', 'required' => true],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/author', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'set_author'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'mode' => ['type' => 'string', 'required' => true],
					'name' => ['type' => 'string', 'required' => false],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/update-module/(?P<module_id>[a-z_]+)', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'update_module'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'data' => ['type' => 'object', 'required' => true],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/generate', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'generate_all'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'only_missing' => ['type' => 'boolean', 'required' => false],
					'skip_edited' => ['type' => 'boolean', 'required' => false],
				],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/generate-module/(?P<module_id>[a-z_]+)', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'generate_module'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/check', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'check_post'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/structured-preview', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'structured_preview'],
				'permission_callback' => [self::class, 'can_edit_post'],
			],
		]);

		register_rest_route(Routes::REST_NAMESPACE, '/post/(?P<id>\\d+)/apply-slug', [
			[
				'methods' => 'POST',
				'callback' => [self::class, 'apply_slug'],
				'permission_callback' => [self::class, 'can_edit_post'],
				'args' => [
					'slug' => ['type' => 'string', 'required' => true],
					'ack_published' => ['type' => 'boolean', 'required' => false],
				],
			],
		]);
	}

	public static function can_edit_post(\WP_REST_Request $request): bool
	{
		$post_id = (int)$request['id'];
		return $post_id > 0 && current_user_can('edit_post', $post_id);
	}

	public static function get_meta(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$post = get_post($post_id);
		if (!$post) {
			return new \WP_Error('cr_post_not_found', '文章不存在或已删除。', ['status' => 404]);
		}
		$meta = MetaStore::get($post_id);
		$options = Settings::get_options();

		return rest_ensure_response([
			'post_id' => $post_id,
			'post' => [
				'slug' => (string)$post->post_name,
				'status' => (string)$post->post_status,
				'permalink' => (string)get_permalink($post),
			],
			'meta' => $meta,
			'options' => [
				'site' => $options['site'] ?? [],
				'modules_order' => $options['modules_order'] ?? [],
				'modules' => $options['modules'] ?? [],
				'page_types' => $options['page_types'] ?? [],
				'structured_output' => $options['structured_output'] ?? [],
				'capabilities' => Capabilities::client_caps($options),
			],
		]);
	}

	public static function set_page_type(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$page_type = sanitize_key((string)$request->get_param('page_type'));
		if ($page_type === '') {
			return new \WP_Error('cr_invalid_page_type', '页面类型无效。', ['status' => 400]);
		}

		$options = Settings::get_options();
		if (!PageTypesConfig::is_known_page_type($page_type, $options)) {
			return new \WP_Error('cr_unknown_page_type', '页面类型不存在，请先在设置页配置。', ['status' => 400]);
		}

		$meta = MetaStore::get($post_id);
		$meta['page_type'] = $page_type;
		MetaStore::update($post_id, $meta);

		return self::get_meta($request);
	}

	public static function set_author(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_AUTHOR_STRATEGY, $options)) {
			return new \WP_Error('cr_pro_required', '该功能仅 Pro 可用：作者策略。', ['status' => 403]);
		}

		$mode = sanitize_key((string)$request->get_param('mode'));
		if (!in_array($mode, ['inherit', 'wp', 'brand'], true)) {
			return new \WP_Error('cr_invalid_author_mode', '作者策略无效。', ['status' => 400]);
		}

		$name = sanitize_text_field((string)$request->get_param('name'));

		$meta = MetaStore::get($post_id);
		$meta['author'] = [
			'mode' => $mode,
			'name' => $name,
		];
		MetaStore::update($post_id, $meta);

		return self::get_meta($request);
	}

	public static function update_module(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$module_id = sanitize_key((string)$request['module_id']);
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return new \WP_Error('cr_invalid_module', '模块无效。', ['status' => 400]);
		}

		$options = Settings::get_options();
		if (!Capabilities::is_module_available($module_id, $options)) {
			return new \WP_Error('cr_pro_required', '该模块仅 Pro 可用。', ['status' => 403]);
		}
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$generator = new V1Generator($options);
		$meta = MetaStore::get($post_id);
		$target_ids = $generator->get_target_module_ids($meta);
		if (!in_array($module_id, $target_ids, true)) {
			return new \WP_Error('cr_module_not_in_page_type', '该模块不属于当前页面类型的模块组合。', ['status' => 400]);
		}

		$data = $request->get_param('data');
		$clean = Sanitizer::sanitize_module_data($module_id, is_array($data) ? $data : []);
		MetaStore::touch_edited($post_id, $module_id, $clean);
		self::maybe_write_summary_excerpt($post_id, $modules_cfg, $module_id, $clean);

		return self::get_meta($request);
	}

	public static function generate_all(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$only_missing = (bool)$request->get_param('only_missing');
		$skip_edited_param = $request->get_param('skip_edited');
		$skip_edited = $skip_edited_param === null ? true : (bool)$skip_edited_param;
		$options = Settings::get_options();
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$generator = new V1Generator($options);

		$meta = MetaStore::get($post_id);

		$needs_ai = false;
		foreach ($generator->get_target_module_ids($meta) as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			if (!Capabilities::is_module_available($module_id, $options)) {
				continue;
			}
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}
			if ($generator->requires_ai($module_id)) {
				$needs_ai = true;
				break;
			}
		}

		if ($needs_ai && !$generator->is_ai_configured()) {
			return new \WP_Error('cr_ai_not_configured', 'AI 未配置：请到设置页填写 AI 接口地址与默认模型。', ['status' => 400]);
		}

		// 收集生成结果
		$generated = [];
		$skipped = [];
		$failed = [];

		foreach ($generator->get_target_module_ids($meta) as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			if (!Capabilities::is_module_available($module_id, $options)) {
				continue;
			}
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}
			if ($only_missing && !self::is_missing($meta['modules'][$module_id]['data'] ?? null)) {
				continue;
			}
			if ($skip_edited) {
				$m = is_array($meta['modules'][$module_id] ?? null) ? $meta['modules'][$module_id] : [];
				$gen_at = (int)($m['generated_at'] ?? 0);
				$edit_at = (int)($m['edited_at'] ?? 0);
				if ($edit_at > 0 && $gen_at > 0 && $edit_at > $gen_at) {
					$label = (string)($cfg['label'] ?? $module_id);
					$skipped[] = ['id' => $module_id, 'label' => $label, 'reason' => '已手动编辑'];
					continue;
				}
			}
			$data = $generator->generate_module($post_id, $module_id, $meta);

			// 处理跳过的模块
			if (!empty($data['_skipped'])) {
				$label = (string)($cfg['label'] ?? $module_id);
				$reason = (string)($data['_reason'] ?? '自适应规则');
				$skipped[] = ['id' => $module_id, 'label' => $label, 'reason' => $reason];
				continue;
			}

			if ($data === []) {
				$detail = $generator->get_ai_last_error();
				$failed[] = ['id' => $module_id, 'error' => $detail];
				continue;
			}

			$clean = Sanitizer::sanitize_module_data($module_id, $data);
			$meta = MetaStore::touch_generated($post_id, $module_id, $clean);
			self::maybe_write_summary_excerpt($post_id, $modules_cfg, $module_id, $clean);
			$generated[] = $module_id;
		}

		// 构建响应消息
		$response = self::get_meta($request);
		if (is_wp_error($response)) {
			return $response;
		}

		$response_data = $response->get_data();
		$response_data['generation_result'] = [
			'generated' => $generated,
			'skipped'   => $skipped,
			'failed'    => $failed,
		];

		// 构建友好提示
		$message_parts = [];
		if (count($generated) > 0) {
			$message_parts[] = sprintf('已生成 %d 个模块', count($generated));
		}
		if (count($skipped) > 0) {
			$skip_labels = array_map(fn($s) => $s['label'], $skipped);
			$message_parts[] = sprintf('跳过 %d 个模块（%s）', count($skipped), implode('、', $skip_labels));
		}
		if (count($failed) > 0) {
			$message_parts[] = sprintf('%d 个模块生成失败', count($failed));
		}

		if (empty($message_parts)) {
			$response_data['generation_message'] = '没有需要生成的模块。';
		} else {
			$response_data['generation_message'] = implode('；', $message_parts) . '。';
		}

		// 如果有模块被跳过，添加提示
		if (count($skipped) > 0 && count($generated) === 0 && count($failed) === 0) {
			$all_edited = true;
			foreach ($skipped as $s) {
				$reason = is_array($s) ? (string)($s['reason'] ?? '') : '';
				if ($reason !== '已手动编辑') {
					$all_edited = false;
					break;
				}
			}
			if ($all_edited) {
				$response_data['generation_message'] = '没有需要生成的模块：已跳过所有手动编辑模块。';
			} else {
				$response_data['generation_message'] = '正文较短，已跳过部分模块的生成。摘要、锚点等必要模块不受影响。如需强制生成所有已启用模块，请前往设置将「模块调度策略」切换为“关闭”。';
			}
		}

		return rest_ensure_response($response_data);
	}

	public static function generate_module(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$module_id = sanitize_key((string)$request['module_id']);
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return new \WP_Error('cr_invalid_module', '模块无效。', ['status' => 400]);
		}

		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_SINGLE_MODULE_GENERATE, $options)) {
			return new \WP_Error('cr_pro_required', '该功能仅 Pro 可用：单模块生成。', ['status' => 403]);
		}

		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
		if (empty($cfg['enabled'])) {
			return new \WP_Error('cr_module_disabled', '该模块未启用。', ['status' => 400]);
		}
		$generator = new V1Generator($options);
		$meta = MetaStore::get($post_id);
		$target_ids = $generator->get_target_module_ids($meta);
		if (!in_array($module_id, $target_ids, true)) {
			return new \WP_Error('cr_module_not_in_page_type', '该模块不属于当前页面类型的模块组合。', ['status' => 400]);
		}
		if ($generator->requires_ai($module_id) && !$generator->is_ai_configured()) {
			return new \WP_Error('cr_ai_not_configured', 'AI 未配置：请到设置页填写 AI 接口地址与默认模型。', ['status' => 400]);
		}

		$data = $generator->generate_module($post_id, $module_id, $meta);
		if ($data === []) {
			$detail = $generator->get_ai_last_error();
			return new \WP_Error('cr_generate_failed', '生成失败：' . sanitize_text_field($detail !== '' ? $detail : 'empty_result'), ['status' => 500]);
		}
		$clean = Sanitizer::sanitize_module_data($module_id, $data);
		MetaStore::touch_generated($post_id, $module_id, $clean);
		self::maybe_write_summary_excerpt($post_id, $modules_cfg, $module_id, $clean);

		return self::get_meta($request);
	}

	public static function check_post(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		return rest_ensure_response([
			'post_id' => $post_id,
			'result' => Checker::check_post($post_id),
		]);
	}

	public static function structured_preview(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		return rest_ensure_response(StructuredData::preview($post_id));
	}

	public static function apply_slug(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$options = Settings::get_options();
		if (!Capabilities::feature_enabled(Capabilities::FEATURE_APPLY_SLUG, $options)) {
			return new \WP_Error('cr_pro_required', '该功能仅 Pro 可用：一键应用 slug。', ['status' => 403]);
		}

		$post = get_post($post_id);
		if (!$post) {
			return new \WP_Error('cr_post_not_found', '文章不存在或已删除。', ['status' => 404]);
		}

		$raw = (string)$request->get_param('slug');
		$requested = self::sanitize_slug_input($raw);
		if ($requested === '') {
			return new \WP_Error('cr_invalid_slug', 'Slug 无效：请使用英文单词与连字符（a-z0-9-）。', ['status' => 400]);
		}

		$current = (string)$post->post_name;
		if ($requested === $current) {
			return rest_ensure_response([
				'post_id' => $post_id,
				'requested' => $requested,
				'applied' => $current,
				'permalink' => (string)get_permalink($post),
			]);
		}

		$is_published = ((string)$post->post_status) === 'publish';
		$ack = (bool)$request->get_param('ack_published');
		if ($is_published && !$ack) {
			return new \WP_Error(
				'cr_slug_requires_ack',
				'本文已经发布，修改 slug 可能影响 SEO。请确认后再应用。',
				['status' => 400]
			);
		}

		$unique = wp_unique_post_slug($requested, $post_id, (string)$post->post_status, (string)$post->post_type, (int)$post->post_parent);
		$updated = wp_update_post([
			'ID' => $post_id,
			'post_name' => $unique,
		], true);

		if (is_wp_error($updated)) {
			return $updated;
		}

		$updated_post = get_post($post_id);
		return rest_ensure_response([
			'post_id' => $post_id,
			'requested' => $requested,
			'applied' => $updated_post ? (string)$updated_post->post_name : $unique,
			'permalink' => $updated_post ? (string)get_permalink($updated_post) : (string)get_permalink($post_id),
		]);
	}

	private static function is_missing($data): bool
	{
		return !is_array($data) || $data === [];
	}

	private static function maybe_write_summary_excerpt(int $post_id, array $modules_cfg, string $module_id, array $data): void
	{
		if ($module_id !== 'summary') {
			return;
		}

		$cfg = is_array($modules_cfg['summary'] ?? null) ? $modules_cfg['summary'] : [];
		$mode = sanitize_key((string)($cfg['summary_excerpt'] ?? 'off'));
		if (!in_array($mode, ['off', 'force', 'compat'], true)) {
			$mode = 'off';
		}
		if ($mode === 'off') {
			return;
		}

		$text = sanitize_text_field((string)($data['text'] ?? ''));
		$text = trim((string)preg_replace('/\\s+/u', ' ', $text));
		if ($text === '') {
			return;
		}

		$post = get_post($post_id);
		if (!$post) {
			return;
		}

		$current = trim((string)$post->post_excerpt);
		if ($mode === 'compat' && $current !== '') {
			return;
		}

		if ($current === $text) {
			return;
		}

		wp_update_post([
			'ID' => $post_id,
			'post_excerpt' => $text,
		]);
	}

	private static function sanitize_slug_input(string $slug): string
	{
		$slug = sanitize_title($slug);
		$slug = strtolower($slug);
		$slug = preg_replace('/[^a-z0-9-]+/', '-', $slug) ?? '';
		$slug = preg_replace('/-+/', '-', $slug) ?? '';
		$slug = trim($slug, '-');

		if (strlen($slug) > 80) {
			$slug = substr($slug, 0, 80);
			$slug = rtrim($slug, '-');
		}

		return $slug;
	}
}
