<?php

declare(strict_types=1);

namespace ContentReady\Rest;

use ContentReady\Admin\Settings;
use ContentReady\Frontend\StructuredData;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Registry;
use ContentReady\Licensing\Capabilities;

final class AssessmentController
{
	public static function register(): void
	{
		register_rest_route(
			Routes::REST_NAMESPACE,
			'/assessment/post/(?P<id>\\d+)',
			[
				'methods' => \WP_REST_Server::CREATABLE,
				'permission_callback' => [self::class, 'can_assess'],
				'callback' => [self::class, 'assess'],
				'args' => [
					'id' => [
						'required' => true,
						'validate_callback' => static function ($value): bool {
							return is_numeric($value) && (int)$value > 0;
						},
					],
				],
			]
		);
	}

	public static function can_assess(\WP_REST_Request $request): bool
	{
		$post_id = (int)$request['id'];
		return $post_id > 0 && current_user_can('manage_options') && current_user_can('edit_post', $post_id);
	}

	public static function assess(\WP_REST_Request $request)
	{
		$post_id = (int)$request['id'];
		$post = get_post($post_id);
		if (!$post) {
			return new \WP_Error('cr_post_not_found', 'post_not_found', ['status' => 404]);
		}

		if ((string)$post->post_status !== 'publish') {
			return new \WP_Error('cr_post_not_publish', 'post_not_publish', ['status' => 400]);
		}

		$permalink = get_permalink($post_id);
		if (!is_string($permalink) || $permalink === '') {
			return new \WP_Error('cr_permalink_missing', 'permalink_missing', ['status' => 400]);
		}

		$options = Settings::get_options();
		$preview = StructuredData::preview($post_id);
		$blocked_reason = sanitize_key((string)($preview['blocked_reason'] ?? ''));
		$meta = MetaStore::get($post_id);
		$expected_enabled_modules_with_data = self::collect_enabled_modules_with_data($meta, $options);
		$missing_enabled_modules = self::collect_enabled_modules_missing_data($meta, $options);
		$missing_enabled_modules_detail = self::module_details($missing_enabled_modules);
		$geo_modules = self::analyze_geo_modules($meta);

		$out = [
			'post_id' => $post_id,
			'permalink' => $permalink,
			'options' => [
				'inject_enabled' => !empty($options['frontend']['inject_enabled']),
				'structured_enabled' => !empty($options['structured_output']['enabled']),
			],
			'structured_preview' => [
				'would_output' => !empty($preview['would_output']),
				'blocked_reason' => $blocked_reason,
			],
			'jsonld' => [
				'total_scripts' => 0,
				'cr_scripts' => 0,
				'other_scripts' => 0,
				'invalid_json' => 0,
				'types' => [],
			],
			'content' => [
				'found_cr_modules' => false,
				'cr_module_count' => 0,
				'expected_enabled_modules_with_data' => $expected_enabled_modules_with_data,
				'expected_enabled_modules_count' => count($expected_enabled_modules_with_data),
				'missing_enabled_modules' => $missing_enabled_modules,
				'missing_enabled_modules_count' => count($missing_enabled_modules),
				'missing_enabled_modules_detail' => $missing_enabled_modules_detail,
			],
			'geo_modules' => $geo_modules,
			'warnings' => [],
			'suggestions' => [],
		];

		$response = wp_remote_get($permalink, [
			'timeout' => 12,
			'redirection' => 3,
			'headers' => [
				'User-Agent' => 'ContentReady/' . (defined('CR_VERSION') ? CR_VERSION : 'dev') . '; ' . home_url('/'),
			],
		]);

		if (is_wp_error($response)) {
			$out['warnings'][] = '抓取前端页面失败（常见原因：站点 loopback 被禁用 / 被安全插件或 WAF 拦截 / 站点需要登录）：' . sanitize_text_field($response->get_error_message()) . '。建议：先到「工具 → 站点健康」检查 loopback；必要时临时关闭安全插件/防火墙规则后重试。';
			$out['suggestions'] = self::default_suggestions($post_id, $blocked_reason);
			return $out;
		}

		$code = (int)wp_remote_retrieve_response_code($response);
		$body = (string)wp_remote_retrieve_body($response);
		if ($code < 200 || $code >= 300) {
			$out['warnings'][] = '前端页面返回非 2xx：HTTP ' . (string)$code . '（可能被鉴权/防火墙/维护模式拦截，导致检查结果不可信）。';
		}
		if (trim($body) === '') {
			$out['warnings'][] = '前端页面返回空内容（常见原因：安全插件/WAF 拦截，或站点禁用了 loopback 请求）。建议：到「工具 → 站点健康」查看 loopback；或临时停用安全插件/开启白名单后重试。';
			$out['suggestions'] = self::default_suggestions($post_id, $blocked_reason);
			return $out;
		}

		$jsonld = self::analyze_jsonld($body);
		$out['jsonld'] = $jsonld;

		$content = self::analyze_content_injection_with_base($body, $out['content']);
		$out['content'] = $content;

		$inject_enabled = !empty($options['frontend']['inject_enabled']);
		$expected_count = (int)($content['expected_enabled_modules_count'] ?? 0);
		if ($inject_enabled && empty($content['found_cr_modules'])) {
			if ($expected_count <= 0) {
				$out['warnings'][] = '前端未检测到 Content Ready 模块标记：该文章目前没有任何“已生成且启用”的模块（可在编辑器点击「生成 GEO」后再检查）。';
			} else {
				$out['warnings'][] = '设置中已开启前端渲染（基于 the_content），且该文章存在可展示模块，但前端未检测到 Content Ready 模块标记：主题可能未渲染 the_content，或被其他插件覆盖。';
			}
		}

		if (!empty($options['structured_output']['enabled']) && empty($preview['would_output']) && $blocked_reason !== '') {
			$out['warnings'][] = self::format_structured_block_warning($blocked_reason);
		}

		if (($jsonld['other_scripts'] ?? 0) > 0 && ($jsonld['cr_scripts'] ?? 0) > 0) {
			$out['warnings'][] = '检测到“本插件 + 其他来源”同时输出结构化（JSON-LD）：如有重复（同类 schema），建议在 SEO 插件中关闭结构化输出（Schema/JSON-LD），或升级 Pro 使用“强制输出”。';
		}

		$out['suggestions'] = self::default_suggestions($post_id, $blocked_reason);
		return $out;
	}

	private static function default_suggestions(int $post_id, string $blocked_reason): array
	{
		$suggestions = [];

		$edit_url = get_edit_post_link($post_id, 'url');
		if (is_string($edit_url) && $edit_url !== '') {
			$suggestions[] = [
				'label' => '打开该文章编辑器',
				'url' => $edit_url,
			];
		}

		if ($blocked_reason === 'auto_not_checked_yet') {
			$suggestions[] = [
				'label' => '提示：智能兼容需先完成兼容性检查',
				'url' => admin_url('admin.php?page=content-ready-settings&tab=advanced'),
			];
		}

		return array_merge($suggestions, [
			[
				'label' => '打开设置（模块与输出）',
				'url' => admin_url('admin.php?page=content-ready-settings&tab=modules_output'),
			],
			[
				'label' => '打开设置（高级/维护）',
				'url' => admin_url('admin.php?page=content-ready-settings&tab=advanced'),
			],
		]);
	}

	private static function analyze_content_injection(string $html): array
	{
		return self::analyze_content_injection_with_base($html, []);
	}

	private static function analyze_content_injection_with_base(string $html, array $base): array
	{
		$count = preg_match_all('/\\bcr-module\\b/', $html, $m);
		$base['found_cr_modules'] = $count > 0;
		$base['cr_module_count'] = (int)$count;
		return $base;
	}

	private static function analyze_jsonld(string $html): array
	{
		$total = 0;
		$cr = 0;
		$other = 0;
		$invalid = 0;
		$types = [];

		if (preg_match_all('/<script\\b([^>]*)type=[\"\\\']application\\/ld\\+json[\"\\\']([^>]*)>(.*?)<\\/script>/is', $html, $matches, PREG_SET_ORDER)) {
			foreach ($matches as $row) {
				$total++;

				$attrs = strtolower((string)($row[1] ?? '')) . ' ' . strtolower((string)($row[2] ?? ''));
				$is_cr = strpos($attrs, 'data-cr-jsonld') !== false;
				if ($is_cr) {
					$cr++;
				} else {
					$other++;
				}

				$raw = (string)($row[3] ?? '');
				$decoded = json_decode($raw, true);
				if ($decoded === null && trim($raw) !== '' && strtolower(trim($raw)) !== 'null') {
					$invalid++;
					continue;
				}
				self::collect_types($decoded, $types);
			}
		}

		arsort($types);
		return [
			'total_scripts' => $total,
			'cr_scripts' => $cr,
			'other_scripts' => $other,
			'invalid_json' => $invalid,
			'types' => $types,
		];
	}

	private static function collect_types($value, array &$types): void
	{
		if (!is_array($value)) {
			return;
		}

		if (isset($value['@type'])) {
			$t = $value['@type'];
			if (is_string($t) && $t !== '') {
				$types[$t] = (int)($types[$t] ?? 0) + 1;
			} elseif (is_array($t)) {
				foreach ($t as $one) {
					if (is_string($one) && $one !== '') {
						$types[$one] = (int)($types[$one] ?? 0) + 1;
					}
				}
			}
		}

		foreach ($value as $v) {
			self::collect_types($v, $types);
		}
	}

	private static function collect_enabled_modules_with_data(array $meta, array $options): array
	{
		$modules_config = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];

		$ids = [];
		foreach ($modules as $module_id => $module) {
			$module_id = sanitize_key((string)$module_id);
			if ($module_id === '') {
				continue;
			}

			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}

			$data = is_array($module['data'] ?? null) ? $module['data'] : [];
			if (self::has_meaningful_data($data)) {
				$ids[] = $module_id;
			}
		}

		sort($ids);
		return $ids;
	}

	private static function collect_enabled_modules_missing_data(array $meta, array $options): array
	{
		$modules_config = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];

		$ids = [];
		foreach ($modules_config as $module_id => $cfg) {
			$module_id = sanitize_key((string)$module_id);
			if ($module_id === '' || !self::is_geo_content_module($module_id)) {
				continue;
			}

			if (!Capabilities::is_module_available($module_id, $options)) {
				continue;
			}

			$cfg = is_array($cfg) ? $cfg : [];
			if (empty($cfg['enabled'])) {
				continue;
			}

			$module = is_array($modules[$module_id] ?? null) ? $modules[$module_id] : [];
			$data = is_array($module['data'] ?? null) ? $module['data'] : [];
			if (!self::has_meaningful_data($data)) {
				$ids[] = $module_id;
			}
		}

		sort($ids);
		return $ids;
	}

	private static function is_geo_content_module(string $module_id): bool
	{
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return false;
		}
		// 仅编辑器辅助，不作为“前端 GEO 信号”。
		return $module_id !== 'slug_suggestions';
	}

	/**
	 * @param array<int, string> $module_ids
	 * @return array<int, array{id:string,label:string,intro:string}>
	 */
	private static function module_details(array $module_ids): array
	{
		$all = Registry::all();
		$out = [];
		foreach ($module_ids as $id) {
			$id = sanitize_key((string)$id);
			if ($id === '' || !isset($all[$id])) {
				continue;
			}
			$out[] = [
				'id' => $id,
				'label' => (string)($all[$id]['label'] ?? $id),
				'intro' => (string)($all[$id]['intro'] ?? ''),
			];
		}
		return $out;
	}

	/**
	 * 面向“转化/营销”与“站长理解成本”：
	 * 不以“当前设置是否启用”为准，而以“推荐 GEO 内容模块”做一份缺失清单。
	 *
	 * @return array{
	 *   generated_ids: array<int, string>,
	 *   missing_ids: array<int, string>,
	 *   missing_detail: array<int, array{id:string,label:string,intro:string}>
	 * }
	 */
	private static function analyze_geo_modules(array $meta): array
	{
		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];
		$all = Registry::all();

		$generated = [];
		$missing = [];
		foreach ($all as $module_id => $def) {
			$module_id = sanitize_key((string)$module_id);
			if ($module_id === '' || !self::is_geo_content_module($module_id)) {
				continue;
			}

			$module = is_array($modules[$module_id] ?? null) ? $modules[$module_id] : [];
			$data = is_array($module['data'] ?? null) ? $module['data'] : [];
			if (self::has_meaningful_data($data)) {
				$generated[] = $module_id;
			} else {
				$missing[] = $module_id;
			}
		}

		sort($generated);
		sort($missing);

		return [
			'generated_ids' => $generated,
			'missing_ids' => $missing,
			'missing_detail' => self::module_details($missing),
		];
	}

	private static function has_meaningful_data($value): bool
	{
		if (is_string($value)) {
			return trim($value) !== '';
		}
		if (is_int($value) || is_float($value)) {
			return true;
		}
		if (is_bool($value)) {
			return $value === true;
		}
		if (!is_array($value)) {
			return false;
		}

		foreach ($value as $v) {
			if (self::has_meaningful_data($v)) {
				return true;
			}
		}
		return false;
	}

	private static function format_structured_block_warning(string $blocked_reason): string
	{
		switch ($blocked_reason) {
			case 'structured_disabled':
				return '结构化输出已关闭（设置 → 模块与输出）。';
			case 'not_article_page_type':
				return '结构化输出：该内容的页面类型不是 article（当前仅 article 输出）。';
			case 'auto_not_checked_yet':
				return '结构化输出：处于“智能兼容”，正在检查是否与其他结构化重复；完成前暂不输出（可稍后刷新前台；如需立即输出可切换为“强制输出”（Pro））。';
			case 'auto_check_failed':
				return '结构化输出：兼容性检查失败，前端已暂停输出（可先排查抓取失败/安全插件/loopback/WAF）。';
			case 'auto_conflict_detected':
				return '结构化输出：检测到重复结构化输出，前端已暂停输出（可关闭 SEO 插件的结构化输出（Schema/JSON-LD），或切换为“强制输出”（Pro））。';
			default:
				return '结构化输出：前端不会输出（原因：' . $blocked_reason . '）。';
		}
	}
}
