<?php

declare(strict_types=1);

namespace ContentReady\PageTypes;

use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;

final class Config
{
	public static function default_page_types(): array
	{
		return [
			[
				'id' => 'article',
				'label' => '通用文章',
				'modules' => array_values(array_filter(ModulesConfig::default_order(), [Registry::class, 'is_content_module'])),
			],
		];
	}

	public static function normalize_page_types($value): array
	{
		if (!is_array($value) || $value === []) {
			return self::default_page_types();
		}

		$out = [];
		foreach ($value as $row) {
			if (!is_array($row)) {
				continue;
			}
			$id = sanitize_key((string)($row['id'] ?? ''));
			$label = sanitize_text_field((string)($row['label'] ?? ''));
			if ($id === '' || $label === '') {
				continue;
			}
			$modules = ModulesConfig::normalize_order($row['modules'] ?? null);
			$modules = array_values(array_filter($modules, [Registry::class, 'is_content_module']));
			$out[] = [
				'id' => $id,
				'label' => $label,
				'modules' => $modules,
			];
		}

		if ($out === []) {
			return self::default_page_types();
		}

		return $out;
	}

	public static function is_known_page_type(string $id, array $options): bool
	{
		$pts = is_array($options['page_types'] ?? null) ? $options['page_types'] : [];
		foreach ($pts as $row) {
			if (is_array($row) && ($row['id'] ?? null) === $id) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 期望的表单结构：
	 * - page_types[<id>][label]
	 * - page_types[<id>][modules][<module_id>]=1
	 * - page_type_new_id / page_type_new_label / page_type_new_modules[<module_id>]=1
	 * - page_types_delete[]=<id>
	 */
	public static function sanitize_from_post(array $post, array $current_options): array
	{
		$current = self::normalize_page_types($current_options['page_types'] ?? null);
		$content_module_ids = array_values(array_filter(ModulesConfig::default_order(), [Registry::class, 'is_content_module']));

		$delete = isset($post['page_types_delete']) ? (array)$post['page_types_delete'] : [];
		$delete_set = [];
		foreach ($delete as $id) {
			$id = sanitize_key((string)$id);
			if ($id !== '' && $id !== 'article') {
				$delete_set[$id] = true;
			}
		}

		$posted = isset($post['page_types']) && is_array($post['page_types']) ? $post['page_types'] : [];
		$out = [];

		foreach ($current as $row) {
			$id = (string)$row['id'];
			if (isset($delete_set[$id])) {
				continue;
			}

			$incoming = is_array($posted[$id] ?? null) ? $posted[$id] : [];
			$label = sanitize_text_field((string)($incoming['label'] ?? $row['label']));
			if ($label === '') {
				$label = (string)$row['label'];
			}

			$selected = [];
			$mods_in = is_array($incoming['modules'] ?? null) ? $incoming['modules'] : [];
			foreach (array_keys($mods_in) as $mid) {
				$mid = sanitize_key((string)$mid);
				if (Registry::is_content_module($mid)) {
					$selected[$mid] = true;
				}
			}

			$modules = [];
			foreach ($content_module_ids as $mid) {
				if (isset($selected[$mid])) {
					$modules[] = $mid;
				}
			}

			$out[] = [
				'id' => $id,
				'label' => $label,
				'modules' => $modules,
			];
		}

		$new_id = isset($post['page_type_new_id']) ? sanitize_key((string)$post['page_type_new_id']) : '';
		$new_label = isset($post['page_type_new_label']) ? sanitize_text_field((string)$post['page_type_new_label']) : '';
		if ($new_id !== '' && $new_label !== '' && !self::exists_id($new_id, $out)) {
			$mods_in = isset($post['page_type_new_modules']) && is_array($post['page_type_new_modules']) ? $post['page_type_new_modules'] : [];
			$selected = [];
			foreach (array_keys($mods_in) as $mid) {
				$mid = sanitize_key((string)$mid);
				if (Registry::is_content_module($mid)) {
					$selected[$mid] = true;
				}
			}
			$modules = [];
			foreach ($content_module_ids as $mid) {
				if (isset($selected[$mid])) {
					$modules[] = $mid;
				}
			}
			$out[] = [
				'id' => $new_id,
				'label' => $new_label,
				'modules' => $modules,
			];
		}

		if (!self::exists_id('article', $out)) {
			array_unshift($out, self::default_page_types()[0]);
		}

		return $out;
	}

	private static function exists_id(string $id, array $list): bool
	{
		foreach ($list as $row) {
			if (is_array($row) && ($row['id'] ?? null) === $id) {
				return true;
			}
		}
		return false;
	}
}
