<?php

declare(strict_types=1);

namespace ContentReady\Modules;

use ContentReady\Util\Anchors;

final class Sanitizer
{
	public static function sanitize_module_data(string $module_id, array $data): array
	{
		switch ($module_id) {
			case 'summary':
				return [
					'text' => sanitize_text_field((string)($data['text'] ?? '')),
				];
			case 'verdict':
				return [
					'text' => self::sanitize_html((string)($data['text'] ?? '')),
				];
			case 'tldr':
			case 'limitations':
				return [
					'items' => self::sanitize_string_list($data['items'] ?? []),
				];
			case 'faq':
				return [
					'items' => self::sanitize_faq_items($data['items'] ?? []),
				];
			case 'references':
				return [
					'items' => self::sanitize_reference_items($data['items'] ?? []),
				];
			case 'last_updated':
				return [
					'date' => sanitize_text_field((string)($data['date'] ?? '')),
				];
			case 'toc':
				return self::sanitize_toc($data);
			case 'table':
				return self::sanitize_table($data);
			case 'slug_suggestions':
				return [
					'suggestions' => self::sanitize_slug_list($data['suggestions'] ?? []),
				];
			default:
				return [];
		}
	}

	private static function sanitize_string_list($items): array
	{
		if (!is_array($items)) {
			return [];
		}
		$out = [];
		foreach ($items as $item) {
			$s = self::sanitize_html((string)$item);
			if ($s !== '') {
				$out[] = $s;
			}
		}
		return $out;
	}

	private static function sanitize_faq_items($items): array
	{
		if (!is_array($items)) {
			return [];
		}
		$out = [];
		foreach ($items as $item) {
			if (!is_array($item)) {
				continue;
			}
			$q = sanitize_text_field((string)($item['q'] ?? ''));
			$a = self::sanitize_html((string)($item['a'] ?? ''));
			if ($q === '' || $a === '') {
				continue;
			}
			$out[] = ['q' => $q, 'a' => $a];
		}
		return $out;
	}

	private static function sanitize_reference_items($items): array
	{
		if (!is_array($items)) {
			return [];
		}
		$out = [];
		foreach ($items as $item) {
			if (!is_array($item)) {
				continue;
			}
			$title = sanitize_text_field((string)($item['title'] ?? ''));
			$url = esc_url_raw((string)($item['url'] ?? ''));
			if ($title === '' && $url === '') {
				continue;
			}
			$out[] = ['title' => $title, 'url' => $url];
		}
		return $out;
	}

	private static function sanitize_html(string $html): string
	{
		$allowed = [
			'p' => [],
			'br' => [],
			'ul' => [],
			'ol' => [],
			'li' => [],
			'strong' => [],
			'em' => [],
			'code' => [],
			'pre' => [],
			'a' => [
				'href' => true,
			],
		];

		$html = wp_kses($html, $allowed);
		return trim($html);
	}

	private static function sanitize_table($data): array
	{
		$data = is_array($data) ? $data : [];
		$caption = sanitize_text_field((string)($data['caption'] ?? ''));
		$notes = sanitize_text_field((string)($data['notes'] ?? ''));

		$columns_in = is_array($data['columns'] ?? null) ? $data['columns'] : [];
		$columns = [];
		foreach ($columns_in as $c) {
			$s = sanitize_text_field((string)$c);
			if ($s !== '') {
				$columns[] = $s;
			}
			if (count($columns) >= 8) {
				break;
			}
		}

		$rows_in = is_array($data['rows'] ?? null) ? $data['rows'] : [];
		$rows = [];
		foreach ($rows_in as $row) {
			if (!is_array($row)) {
				continue;
			}
			$r = [];
			foreach ($row as $cell) {
				$r[] = sanitize_text_field((string)$cell);
				if (count($r) >= 8) {
					break;
				}
			}
			if ($columns !== [] && count($r) !== count($columns)) {
				continue;
			}
			if ($r !== []) {
				$rows[] = $r;
			}
			if (count($rows) >= 20) {
				break;
			}
		}

		return [
			'caption' => $caption,
			'columns' => $columns,
			'rows' => $rows,
			'notes' => $notes,
		];
	}

	private static function sanitize_toc($data): array
	{
		$data = is_array($data) ? $data : [];
		$items_in = is_array($data['items'] ?? null) ? $data['items'] : [];
		$out = [];

		foreach ($items_in as $it) {
			if (!is_array($it)) {
				continue;
			}
			$id = Anchors::sanitize_anchor_id((string)($it['id'] ?? ''));
			$text = sanitize_text_field((string)($it['text'] ?? ''));
			$level = (int)($it['level'] ?? 2);
			if ($id === '' || $text === '') {
				continue;
			}
			$level = max(2, min(5, $level));
			$out[] = [
				'id' => $id,
				'text' => $text,
				'level' => $level,
			];
			if (count($out) >= 200) {
				break;
			}
		}

		return [
			'items' => $out,
		];
	}

	private static function sanitize_slug_list($items): array
	{
		if (!is_array($items)) {
			return [];
		}

		$out = [];
		foreach ($items as $it) {
			$s = sanitize_title((string)$it);
			$s = preg_replace('/[^a-z0-9-]+/', '', strtolower($s)) ?? '';
			$s = trim($s, '-');
			if ($s === '') {
				continue;
			}
			if (strlen($s) > 80) {
				$s = substr($s, 0, 80);
				$s = rtrim($s, '-');
			}
			if (!in_array($s, $out, true)) {
				$out[] = $s;
			}
			if (count($out) >= 5) {
				break;
			}
		}

		return $out;
	}
}

