<?php

declare(strict_types=1);

namespace ContentReady\Meta;

use ContentReady\Licensing\DailyQuota;
use ContentReady\Modules\Registry;
use ContentReady\Util\Anchors;

final class MetaStore
{
	public const META_KEY = 'cr_meta_v1';
	public const META_VERSION = 1;

	public static function get(int $post_id): array
	{
		$raw = get_post_meta($post_id, self::META_KEY, true);
		$meta = is_array($raw) ? $raw : [];

		return self::normalize($meta);
	}

	public static function update(int $post_id, array $meta): void
	{
		$normalized = self::normalize($meta);
		update_post_meta($post_id, self::META_KEY, $normalized);
		/**
		 * 内部钩子：当本插件 meta 更新后触发（用于兼容自检等异步任务调度）。
		 *
		 * @param int   $post_id
		 * @param array $meta 归一化后的 meta（可直接读写回 MetaStore::update）。
		 */
		do_action('cr_meta_updated', $post_id, $normalized);
	}

	public static function touch_generated(int $post_id, string $module_id, array $data): array
	{
		$now = time();
		$meta = self::get($post_id);
		$meta['generated_at'] = $now;
		$meta['edited_at'] = $now;

		$meta['modules'][$module_id] = [
			'generated_at' => $now,
			'edited_at' => $now,
			'data' => $data,
		];

		self::update($post_id, $meta);
		DailyQuota::charge_if_needed($post_id, $module_id);
		return $meta;
	}

	public static function touch_edited(int $post_id, string $module_id, array $data): array
	{
		$now = time();
		$meta = self::get($post_id);
		$meta['edited_at'] = $now;

		$module = $meta['modules'][$module_id] ?? [
			'generated_at' => 0,
			'edited_at' => 0,
			'data' => [],
		];

		$module['edited_at'] = $now;
		$module['data'] = $data;
		$meta['modules'][$module_id] = $module;

		self::update($post_id, $meta);
		return $meta;
	}

	public static function normalize(array $meta): array
	{
		$meta_version = (int)($meta['meta_version'] ?? 0);
		if ($meta_version <= 0) {
			$meta_version = self::META_VERSION;
		}

		$normalized = [
			'meta_version' => $meta_version,
			'generated_at' => (int)($meta['generated_at'] ?? 0),
			'edited_at' => (int)($meta['edited_at'] ?? 0),
			'page_type' => is_string($meta['page_type'] ?? null) ? (string)$meta['page_type'] : 'article',
			'author' => [
				'mode' => 'inherit', // inherit | wp | brand
				'name' => '',
			],
			'anchors' => [
				'modules' => Anchors::normalize_module_anchors((is_array($meta['anchors'] ?? null) ? $meta['anchors']['modules'] ?? null : null)),
				'headings' => Anchors::normalize_heading_anchors((is_array($meta['anchors'] ?? null) ? $meta['anchors']['headings'] ?? null : null)),
			],
			'checks' => is_array($meta['checks'] ?? null) ? $meta['checks'] : [],
			'modules' => [],
			'compat' => is_array($meta['compat'] ?? null) ? $meta['compat'] : [],
		];

		$author = is_array($meta['author'] ?? null) ? $meta['author'] : [];
		$mode = sanitize_key((string)($author['mode'] ?? 'inherit'));
		if (!in_array($mode, ['inherit', 'wp', 'brand'], true)) {
			$mode = 'inherit';
		}
		$normalized['author'] = [
			'mode' => $mode,
			'name' => sanitize_text_field((string)($author['name'] ?? '')),
		];

		$modules = is_array($meta['modules'] ?? null) ? $meta['modules'] : [];
		foreach (Registry::all_ids() as $module_id) {
			$module = is_array($modules[$module_id] ?? null) ? $modules[$module_id] : [];
			$normalized['modules'][$module_id] = [
				'generated_at' => (int)($module['generated_at'] ?? 0),
				'edited_at' => (int)($module['edited_at'] ?? 0),
				'data' => is_array($module['data'] ?? null) ? $module['data'] : [],
			];
		}

		return $normalized;
	}
}

