<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

use ContentReady\Admin\Settings;
use ContentReady\Rest\Routes;

/**
 * 每日配额：对“生成”类 REST 入口做统一拦截，避免在大文件 PostController 里堆门禁逻辑。
 */
final class QuotaRuntime
{
	public static function register(): void
	{
		add_filter('rest_pre_dispatch', [self::class, 'maybe_block_generation'], 20, 3);
	}

	/**
	 * @param mixed $result
	 * @return mixed
	 */
	public static function maybe_block_generation($result, \WP_REST_Server $server, \WP_REST_Request $request)
	{
		if ($result !== null) {
			return $result;
		}

		$route = (string)$request->get_route();
		if ($route === '' || strpos($route, '/' . Routes::REST_NAMESPACE . '/') !== 0) {
			return $result;
		}

		$method = strtoupper((string)$request->get_method());
		if ($method !== 'POST') {
			return $result;
		}

		$is_generate_all = strpos($route, '/generate') !== false && strpos($route, '/generate-module/') === false;
		$is_generate_module = strpos($route, '/generate-module/') !== false;
		if (!$is_generate_all && !$is_generate_module) {
			return $result;
		}

		$post_id = (int)$request->get_param('id');
		if ($post_id <= 0) {
			return $result;
		}

		if ($is_generate_module) {
			$module_id = sanitize_key((string)$request->get_param('module_id'));
			if ($module_id === 'last_updated') {
				return $result;
			}
		}

		$options = Settings::get_options();
		$check = DailyQuota::can_generate_post($post_id, $options);
		if (!empty($check['ok'])) {
			return $result;
		}

		return new \WP_Error('cr_quota_exceeded', (string)($check['message'] ?? '已达今日配额。'), ['status' => 429]);
	}
}

