<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

use ContentReady\Admin\Settings;
use ContentReady\Rest\Routes;

final class LicenseRuntime
{
	private const DAILY_REFRESH_HOOK = 'cr_license_daily_refresh';

	public static function register(): void
	{
		add_action('admin_init', [self::class, 'maybe_refresh_admin'], 20);
		add_filter('rest_pre_dispatch', [self::class, 'maybe_refresh_rest'], 5, 3);

		add_action('init', [self::class, 'maybe_schedule_daily_refresh'], 20);
		add_action(self::DAILY_REFRESH_HOOK, [self::class, 'run_daily_refresh'], 10);
	}

	public static function maybe_refresh_admin(): void
	{
		if (!current_user_can('manage_options')) {
			return;
		}
		LicenseGate::refresh_and_persist(false, 'admin');
	}

	/**
	 * 对 Pro 敏感的 REST 操作：强制刷新一次，确保“禁用立即生效”。
	 *
	 * @param mixed $result
	 * @return mixed
	 */
	public static function maybe_refresh_rest($result, \WP_REST_Server $server, \WP_REST_Request $request)
	{
		$route = (string)$request->get_route();
		if ($route === '' || strpos($route, '/' . Routes::REST_NAMESPACE . '/') !== 0) {
			return $result;
		}

		$method = strtoupper((string)$request->get_method());
		$force = false;
		if ($method === 'POST') {
			if (strpos($route, '/generate-module/') !== false) {
				$force = true;
			} elseif (strpos($route, '/generate') !== false) {
				// 生成（可能包含 Pro 模块）：强制刷新一次，确保“禁用立即生效”。
				$force = true;
			} elseif (strpos($route, '/apply-slug') !== false) {
				$force = true;
			} elseif (strpos($route, '/author') !== false) {
				// 作者策略为 Pro：强制刷新一次，确保“禁用立即生效”。
				$force = true;
			} elseif (strpos($route, '/update-module/') !== false) {
				$module_id = sanitize_key((string)$request->get_param('module_id'));
				if ($module_id !== '' && Capabilities::module_plan($module_id) === Capabilities::PLAN_PRO) {
					$force = true;
				}
			}
		}

		LicenseGate::refresh_and_persist($force, $force ? 'rest_force' : 'rest');

		return $result;
	}

	public static function maybe_schedule_daily_refresh(): void
	{
		$options = Settings::get_raw_options();
		$license = is_array($options['license'] ?? null) ? $options['license'] : [];
		$key = trim((string)($license['key'] ?? ''));

		if ($key === '') {
			wp_clear_scheduled_hook(self::DAILY_REFRESH_HOOK);
			return;
		}

		$next = self::next_daily_refresh_timestamp();
		$current = wp_next_scheduled(self::DAILY_REFRESH_HOOK);
		if ($current) {
			if (abs((int)$current - (int)$next) > 60) {
				wp_unschedule_event((int)$current, self::DAILY_REFRESH_HOOK);
				wp_schedule_event((int)$next, 'daily', self::DAILY_REFRESH_HOOK);
			}
			return;
		}

		wp_schedule_event((int)$next, 'daily', self::DAILY_REFRESH_HOOK);
	}

	public static function run_daily_refresh(): void
	{
		LicenseGate::refresh_and_persist(true, 'cron_daily');
	}

	private static function next_daily_refresh_timestamp(): int
	{
		$time = (string)apply_filters('content_ready_license_daily_refresh_time', '05:00');
		$time = trim($time);
		if (!preg_match('/^(\\d{1,2}):(\\d{2})$/', $time, $m)) {
			$time = '05:00';
			$m = ['05:00', '05', '00'];
		}

		$hour = max(0, min(23, (int)$m[1]));
		$min = max(0, min(59, (int)$m[2]));

		$tz = self::daily_refresh_timezone();
		$now = new \DateTime('now', $tz);
		$run = (clone $now)->setTime($hour, $min, 0);
		if ($run <= $now) {
			$run = $run->modify('+1 day');
		}
		return (int)$run->format('U');
	}

	private static function daily_refresh_timezone(): \DateTimeZone
	{
		$tz = (string)apply_filters('content_ready_license_daily_refresh_timezone', 'Asia/Shanghai');
		$tz = trim($tz);
		if ($tz === '') {
			$tz = 'Asia/Shanghai';
		}

		try {
			return new \DateTimeZone($tz);
		} catch (\Exception $e) {
			return new \DateTimeZone('UTC');
		}
	}
}
