<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

use ContentReady\Modules\Registry;

/**
 * 每日篇数限制/配额（本地计数 + 服务端下发限额）。
 *
 * 约定：
 * - 限额来自在线校验（license.verify 的 quota_daily_limit），0 表示不限制。
 * - 计数以“当天首次为某篇文章生成 AI 模块”为准（同一文章同一天多次生成只计 1 篇）。
 */
final class DailyQuota
{
	private const STATE_OPTION_KEY = 'cr_quota_state_v1';
	private const POST_META_DAY_KEY = 'cr_quota_day_v1';
	private const AUTOMATION_NOTICE_KEY = 'cr_automation_block_notice_v1';

	private static function day_key(): string
	{
		$ts = (int)current_time('timestamp');
		if (function_exists('wp_date')) {
			return (string)wp_date('Ymd', $ts);
		}
		return (string)date_i18n('Ymd', $ts);
	}

	public static function seconds_until_reset(): int
	{
		$tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone('UTC');
		$now = new \DateTime('now', $tz);
		$reset = (clone $now)->setTime(0, 0, 0)->modify('+1 day');
		$diff = (int)$reset->format('U') - (int)$now->format('U');
		return max(60, min(DAY_IN_SECONDS, $diff));
	}

	/**
	 * @return array{day:string,used:int}
	 */
	public static function state(): array
	{
		$today = self::day_key();
		$raw = get_option(self::STATE_OPTION_KEY, []);
		$raw = is_array($raw) ? $raw : [];

		$day = sanitize_text_field((string)($raw['day'] ?? ''));
		$used = (int)($raw['used'] ?? 0);
		if ($day !== $today) {
			$day = $today;
			$used = 0;
			update_option(self::STATE_OPTION_KEY, ['day' => $day, 'used' => 0], false);
		}

		return [
			'day' => $day,
			'used' => max(0, $used),
		];
	}

	/**
	 * 0 表示不限制。
	 */
	public static function limit(array $options): int
	{
		$license = is_array($options['license'] ?? null) ? $options['license'] : [];
		$limit = (int)($license['quota_daily_limit'] ?? 0);
		$limit = (int)apply_filters('content_ready_quota_daily_limit', $limit, $options);
		return max(0, min(100000, $limit));
	}

	/**
	 * @return array{ok:bool,used:int,limit:int,reason:string,message:string}
	 */
	public static function can_generate_post(int $post_id, array $options): array
	{
		$limit = self::limit($options);
		$state = self::state();
		$used = (int)$state['used'];
		$day = (string)$state['day'];

		if ($limit <= 0) {
			return [
				'ok' => true,
				'used' => $used,
				'limit' => $limit,
				'reason' => '',
				'message' => '',
			];
		}

		$charged_day = sanitize_text_field((string)get_post_meta($post_id, self::POST_META_DAY_KEY, true));
		if ($charged_day === $day) {
			return [
				'ok' => true,
				'used' => $used,
				'limit' => $limit,
				'reason' => '',
				'message' => '',
			];
		}

		if ($used < $limit) {
			return [
				'ok' => true,
				'used' => $used,
				'limit' => $limit,
				'reason' => '',
				'message' => '',
			];
		}

		$plan = Capabilities::plan($options);
		$msg = sprintf(
			'已达今日配额：已生成 %d 篇 / 上限 %d 篇（%s）。配额将于次日自动重置。',
			$used,
			$limit,
			$plan
		);

		return [
			'ok' => false,
			'used' => $used,
			'limit' => $limit,
			'reason' => 'quota_exceeded',
			'message' => $msg,
		];
	}

	/**
	 * 当某模块成功写入“生成结果”时，为文章计入一次配额。
	 */
	public static function charge_if_needed(int $post_id, string $module_id): void
	{
		if ($post_id <= 0) {
			return;
		}

		$module_id = sanitize_key($module_id);
		if ($module_id === '' || in_array($module_id, ['last_updated', 'toc'], true)) {
			return;
		}
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return;
		}

		$day = self::day_key();
		$charged_day = sanitize_text_field((string)get_post_meta($post_id, self::POST_META_DAY_KEY, true));
		if ($charged_day === $day) {
			return;
		}

		// 先写入 post_meta，避免 option 更新失败导致的重复计数。
		update_post_meta($post_id, self::POST_META_DAY_KEY, $day);

		$state = self::state();
		$used = (int)$state['used'];
		$used++;

		update_option(self::STATE_OPTION_KEY, [
			'day' => $day,
			'used' => $used,
			'updated_at' => time(),
		], false);
	}

	public static function record_automation_block(string $context, string $message): void
	{
		update_option(self::AUTOMATION_NOTICE_KEY, [
			'at' => time(),
			'seen_at' => 0,
			'context' => sanitize_key($context),
			'reason' => 'quota_exceeded',
			'message' => sanitize_text_field($message),
		], false);
	}
}
