<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

use ContentReady\Modules\Registry;

/**
 * 能力矩阵（SSOT）：统一定义 Free/Pro 的模块与功能范围。
 *
 * 说明：
 * - AI 永远由用户自配（不属于 Pro 能力）。
 * - 当前阶段：License 在线校验未落地，暂以“填了 license key”视为 Pro（后续替换为服务端校验）。
 */
final class Capabilities
{
	public const PLAN_FREE = 'free';
	public const PLAN_PRO = 'pro';
	public const PLAN_MAX = 'max';

	/**
	 * Free 允许的内容模块（其余内容模块为 Pro）。
	 * @var array<string, bool>
	 */
	private const FREE_CONTENT_MODULES = [
		'summary' => true,
		'toc' => true,
		'faq' => true,
		'last_updated' => true,
		'slug_suggestions' => true,
	];

	/**
	 * Pro 功能点（非模块）的能力 key。
	 */
	public const FEATURE_SINGLE_MODULE_GENERATE = 'single_module_generate';
	public const FEATURE_APPLY_SLUG = 'apply_slug';
	public const FEATURE_AUTHOR_STRATEGY = 'author_strategy';
	public const FEATURE_STRUCTURED_FORCE = 'structured_force';
	public const FEATURE_AUTO_GEO_ON_PUBLISH = 'auto_geo_on_publish';
	public const FEATURE_BACKFILL = 'backfill';
	public const FEATURE_ADAPTIVE_GENERATION = 'adaptive_generation';
	public const FEATURE_LIGHTWEIGHT_DISPATCH = 'lightweight_dispatch';

	/**
	 * 功能点（非模块）定义（SSOT）。
	 *
	 * @return array<string, array{group:string,label:string,description:string,pro_only:bool}>
	 */
	public static function feature_definitions(): array
	{
		return [
			self::FEATURE_SINGLE_MODULE_GENERATE => [
				'group' => '编辑器与效率',
				'label' => '单模块生成',
				'description' => '只生成当前模块，方便针对性补齐与微调。',
				'pro_only' => true,
			],
			self::FEATURE_APPLY_SLUG => [
				'group' => '编辑器与效率',
				'label' => 'Slug 推荐与一键应用',
				'description' => '生成更友好的链接别名，并提供二次确认与冲突处理。',
				'pro_only' => true,
			],
			self::FEATURE_ADAPTIVE_GENERATION => [
				'group' => '生成与自动化',
				'label' => '自适应模块调度',
				'description' => '根据正文信号自动跳过不适合的模块，避免结构膨胀并节省 Token。',
				'pro_only' => true,
			],
			self::FEATURE_LIGHTWEIGHT_DISPATCH => [
				'group' => '生成与自动化',
				'label' => '固定轻量化模块调度',
				'description' => '使用推荐模块组合减少 AI 调用次数，优先保证通用收益与成本可控。',
				'pro_only' => true,
			],
			self::FEATURE_AUTO_GEO_ON_PUBLISH => [
				'group' => '生成与自动化',
				'label' => '发布时自动生成 GEO',
				'description' => '文章发布时自动生成/补齐模块（异步执行）。',
				'pro_only' => true,
			],
			self::FEATURE_BACKFILL => [
				'group' => '生成与自动化',
				'label' => '老文章回填',
				'description' => '按计划批量为历史文章生成/补齐模块。',
				'pro_only' => true,
			],
			self::FEATURE_STRUCTURED_FORCE => [
				'group' => '结构化输出',
				'label' => '结构化输出模式（强制输出/关闭）',
				'description' => '当检测到站点已有其他结构化输出时，允许切换为“强制输出”或“关闭”（默认智能兼容）。',
				'pro_only' => true,
			],
			self::FEATURE_AUTHOR_STRATEGY => [
				'group' => '结构化输出',
				'label' => '作者策略（品牌作者）',
				'description' => '结构化输出可使用“品牌作者”而不是 WordPress 作者。',
				'pro_only' => true,
			],
		];
	}

	public static function is_pro(array $options): bool
	{
		$license = is_array($options['license'] ?? null) ? $options['license'] : [];
		$now = time();

		$status = sanitize_key((string)($license['status'] ?? 'unlicensed'));
		$plan = sanitize_key((string)($license['plan'] ?? self::PLAN_FREE));
		$expires_at = (int)($license['expires_at'] ?? 0);
		$grace_until = (int)($license['grace_until'] ?? 0);

		if (!in_array($plan, [self::PLAN_PRO, self::PLAN_MAX], true)) {
			return false;
		}

		if ($status === 'active' && $expires_at > $now) {
			return true;
		}

		// 网络故障宽限：仅当之前校验成功过，且仍在宽限期内。
		if ($status === 'grace' && $grace_until > $now) {
			return true;
		}

		return false;
	}

	public static function plan(array $options): string
	{
		if (!self::is_pro($options)) {
			return self::PLAN_FREE;
		}
		$license = is_array($options['license'] ?? null) ? $options['license'] : [];
		$plan = sanitize_key((string)($license['plan'] ?? self::PLAN_PRO));
		return in_array($plan, [self::PLAN_PRO, self::PLAN_MAX], true) ? $plan : self::PLAN_PRO;
	}

	public static function is_module_available(string $module_id, array $options): bool
	{
		if (!Registry::is_valid_id($module_id)) {
			return false;
		}
		if (!Registry::is_content_module($module_id)) {
			// 代码层模块默认 Free 可用（如 anchors / structured_output）。
			return true;
		}

		if (isset(self::FREE_CONTENT_MODULES[$module_id])) {
			return true;
		}

		return self::is_pro($options);
	}

	public static function module_plan(string $module_id): string
	{
		if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
			return self::PLAN_FREE;
		}
		return isset(self::FREE_CONTENT_MODULES[$module_id]) ? self::PLAN_FREE : self::PLAN_PRO;
	}

	public static function feature_enabled(string $feature, array $options): bool
	{
		$defs = self::feature_definitions();
		if (!isset($defs[$feature])) {
			return true;
		}

		if (empty($defs[$feature]['pro_only'])) {
			return true;
		}

		return self::is_pro($options);
	}

	/**
	 * 对 options 进行“计划约束”归一化，避免 Free 通过手工 POST/篡改配置启用 Pro 能力。
	 */
	public static function normalize_options(array $options): array
	{
		$is_pro = self::is_pro($options);

		// 结构化输出：Free 仅允许 auto（仍允许启用/停用）。
		if (!$is_pro) {
			if (!is_array($options['structured_output'] ?? null)) {
				$options['structured_output'] = [];
			}
			$options['structured_output']['mode'] = 'auto';
		}

		// 隐私发送范围：Free 不允许 selected_modules（细粒度）。
		if (!$is_pro) {
			if (!is_array($options['privacy'] ?? null)) {
				$options['privacy'] = [];
			}
			$scope = (string)($options['privacy']['send_scope'] ?? 'full');
			if ($scope === 'selected_modules') {
				$options['privacy']['send_scope'] = 'title_summary';
			}
		}

		// 模块配置：Free 强制关闭 Pro 内容模块。
		$modules = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		if (!$is_pro && is_array($modules)) {
			foreach (array_keys($modules) as $mid) {
				$mid = sanitize_key((string)$mid);
				if (!Registry::is_valid_id($mid) || !Registry::is_content_module($mid)) {
					continue;
				}
				if (isset(self::FREE_CONTENT_MODULES[$mid])) {
					continue;
				}
				$cfg = is_array($modules[$mid] ?? null) ? $modules[$mid] : [];
				$cfg['enabled'] = false;
				$cfg['allow_ai'] = false;
				$modules[$mid] = $cfg;
			}
			$options['modules'] = $modules;
		}

		// 页面类型：Free 仅保留 Free 内容模块组合。
		if (!$is_pro) {
			$page_types = is_array($options['page_types'] ?? null) ? $options['page_types'] : [];
			$out = [];
			foreach ($page_types as $row) {
				if (!is_array($row)) {
					continue;
				}
				$mods = is_array($row['modules'] ?? null) ? $row['modules'] : [];
				$filtered = [];
				foreach ($mods as $m) {
					$m = sanitize_key((string)$m);
					if (isset(self::FREE_CONTENT_MODULES[$m])) {
						$filtered[] = $m;
					}
				}
				$row['modules'] = $filtered;
				$out[] = $row;
			}
			$options['page_types'] = $out;
		}

		// 自动化：Free 强制关闭。
		if (!$is_pro) {
			if (!is_array($options['generation'] ?? null)) {
				$options['generation'] = [];
			}
			$options['generation']['auto_geo_on_publish'] = false;
			$options['generation']['backfill_enabled'] = false;
			$options['generation']['adaptive_enabled'] = false;
			$options['generation']['dispatch_mode'] = 'off';
		}

		// 作者策略：Free 强制使用 WordPress 作者（站点层）。
		if (!$is_pro) {
			if (!is_array($options['site'] ?? null)) {
				$options['site'] = [];
			}
			$options['site']['author_mode'] = 'wp';
			$options['site']['brand_author_name'] = '';
		}

		return $options;
	}

	/**
	 * 提供给前端脚本/REST 的最小能力集（不包含敏感信息）。
	 */
	public static function client_caps(array $options): array
	{
		$plan = self::plan($options);
		$is_pro = $plan !== self::PLAN_FREE;

		$module_plans = [];
		foreach (Registry::all_ids() as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			$module_plans[$module_id] = self::module_plan($module_id);
		}

		$features = [];
		foreach (self::feature_definitions() as $key => $def) {
			$features[$key] = !empty($def['pro_only']) ? $is_pro : true;
		}

		return [
			'plan' => $plan,
			'features' => $features,
			'modules' => $module_plans,
		];
	}
}
