<?php

declare(strict_types=1);

namespace ContentReady\Licensing;

use ContentReady\Admin\Settings;

/**
 * Pro 自动化统一门禁（在线票据，fail-closed）。
 *
 * 目标：
 * - 自动化能力（发文自动生成/回填等）执行前必须拿到“短 TTL 票据”
 * - 拿不到票据时：直接跳过，并在后台给出提示（避免离线无限跑）
 */
final class AutomationGate
{
	private const TICKET_OPTION_KEY = 'cr_pro_ticket_v1';
	private const NOTICE_OPTION_KEY = 'cr_automation_block_notice_v1';

	/**
	 * @return bool allowed
	 */
	public static function ensure_ticket(string $context): bool
	{
		$context = sanitize_key($context);
		if ($context === '') {
			$context = 'automation';
		}

		$cached = get_option(self::TICKET_OPTION_KEY, []);
		$cached = is_array($cached) ? $cached : [];
		$ticket = sanitize_text_field((string)($cached['ticket'] ?? ''));
		$expires_at = (int)($cached['expires_at'] ?? 0);
		if ($ticket !== '' && $expires_at - time() > 10) {
			return true;
		}

		$raw = Settings::get_raw_options();
		$license = is_array($raw['license'] ?? null) ? $raw['license'] : [];
		$key = trim((string)($license['key'] ?? ''));
		if ($key === '') {
			self::record_notice($context, 'missing_license_key', self::message_for($context, '未填写 License Key。请先在「许可证」页填写并校验。'));
			self::clear_ticket_cache();
			return false;
		}

		$host = LicenseClient::site_host();
		$site_url = (string)get_site_url();
		$plugin_version = defined('CR_VERSION') ? (string)CR_VERSION : '';

		$result = TicketClient::request($key, $host, $site_url, $plugin_version, $context);
		if (empty($result['ok'])) {
			$err = sanitize_text_field((string)($result['error'] ?? 'request_failed'));
			self::record_notice($context, 'ticket_request_failed', self::message_for($context, '无法获取在线票据：' . $err . '（自动化已跳过）'));
			self::clear_ticket_cache();
			return false;
		}

		$data = is_array($result['data'] ?? null) ? $result['data'] : [];
		$status = sanitize_key((string)($data['status'] ?? ''));
		$plan = sanitize_key((string)($data['plan'] ?? ''));
		$base_domain = sanitize_text_field((string)($data['base_domain'] ?? ''));
		$message = sanitize_text_field((string)($data['message'] ?? ''));
		$server_time = (int)($data['server_time'] ?? 0);

		if ($status !== 'active') {
			self::apply_definitive_license_status($raw, $status, $plan, $base_domain, $message);
			self::record_notice($context, 'ticket_denied', self::message_for($context, self::status_to_human($status, $base_domain)));
			self::clear_ticket_cache();
			return false;
		}

		$ticket = sanitize_text_field((string)($data['ticket'] ?? ''));
		$ticket_ttl_sec = (int)($data['ticket_ttl_sec'] ?? 0);
		$ticket_expires_at = (int)($data['ticket_expires_at'] ?? 0);
		$quota_daily_limit = max(0, min(100000, (int)($data['quota_daily_limit'] ?? 0)));
		$local_expires_at = 0;

		if ($ticket_expires_at > 0 && $server_time > 0 && $ticket_expires_at > $server_time) {
			$local_expires_at = time() + ($ticket_expires_at - $server_time);
		} elseif ($ticket_ttl_sec > 0) {
			$local_expires_at = time() + max(15, min(86400, $ticket_ttl_sec));
		}

		if ($ticket === '' || $local_expires_at - time() <= 10) {
			self::record_notice($context, 'invalid_ticket', self::message_for($context, '票据无效或已过期（自动化已跳过）。'));
			self::clear_ticket_cache();
			return false;
		}

		self::apply_active_license_status($raw, $plan, $base_domain, $message, $quota_daily_limit, $local_expires_at);

		update_option(self::TICKET_OPTION_KEY, [
			'ticket' => $ticket,
			'expires_at' => $local_expires_at,
			'checked_at' => time(),
		], false);

		return true;
	}

	private static function clear_ticket_cache(): void
	{
		update_option(self::TICKET_OPTION_KEY, [
			'ticket' => '',
			'expires_at' => 0,
			'checked_at' => time(),
		], false);
	}

	private static function record_notice(string $context, string $reason, string $message): void
	{
		update_option(self::NOTICE_OPTION_KEY, [
			'at' => time(),
			'seen_at' => 0,
			'context' => sanitize_key($context),
			'reason' => sanitize_key($reason),
			'message' => $message,
		], false);
	}

	private static function message_for(string $context, string $body): string
	{
		$label = self::context_label($context);
		return 'Content Ready：' . $label . '未执行（' . $body . '）';
	}

	private static function context_label(string $context): string
	{
		$context = sanitize_key($context);
		switch ($context) {
			case 'automation_auto_geo':
				return '发布时自动生成 GEO';
			case 'automation_backfill':
				return '老文章回填';
			default:
				return '自动化';
		}
	}

	private static function status_to_human(string $status, string $base_domain): string
	{
		switch ($status) {
			case 'disabled':
				return '授权已禁用（自动化已跳过）。';
			case 'invalid_key':
				return 'License Key 无效（自动化已跳过）。';
			case 'invalid_domain':
				return $base_domain !== ''
					? ('域名不匹配（允许域：' . $base_domain . '）（自动化已跳过）。')
					: '域名不匹配（自动化已跳过）。';
			case 'pro_required':
				return '当前授权非 Pro（自动化已跳过）。';
			default:
				return '无法通过在线校验（自动化已跳过）。';
		}
	}

	/**
	 * 票据成功时：同步到本地 license（用于 Pro 能力归一化 + 每日配额）。
	 */
	private static function apply_active_license_status(array $raw, string $plan, string $base_domain, string $message, int $quota_daily_limit, int $expires_at): void
	{
		$license = is_array($raw['license'] ?? null) ? $raw['license'] : [];

		$plan = sanitize_key($plan);
		if (!in_array($plan, ['pro', 'max'], true)) {
			$plan = 'pro';
		}

		$license['checked_at'] = time();
		$license['status'] = 'active';
		$license['plan'] = $plan;
		$license['base_domain'] = strtolower(trim($base_domain));
		$license['last_message'] = $message;
		$license['last_http_status'] = 200;
		$license['last_error'] = '';
		$license['quota_daily_limit'] = max(0, min(100000, $quota_daily_limit));
		$license['last_ok_at'] = time();
		$license['last_ok_plan'] = $plan;
		$license['expires_at'] = max(time() + 60, $expires_at);

		$raw['license'] = $license;
		Settings::update_options($raw);
	}

	/**
	 * 票据接口返回了“确定性”状态时，同步到本地 license（用于禁用立即生效）。
	 */
	private static function apply_definitive_license_status(array $raw, string $status, string $plan, string $base_domain, string $message): void
	{
		if (!in_array($status, ['disabled', 'invalid_key', 'invalid_domain', 'pro_required'], true)) {
			return;
		}

		$license = is_array($raw['license'] ?? null) ? $raw['license'] : [];
		$license['checked_at'] = time();
		$license['base_domain'] = strtolower(trim($base_domain));
		$license['last_message'] = $message;
		$license['last_http_status'] = 200;
		$license['last_error'] = '';

		if ($status === 'pro_required') {
			$license['status'] = 'active';
			$license['plan'] = 'free';
			$license['expires_at'] = time() + 60;
			$raw['license'] = $license;
			Settings::update_options($raw);
			return;
		}

		$license['status'] = $status;
		$license['plan'] = 'free';
		$license['expires_at'] = time() + 60;
		$license['grace_until'] = 0;
		$raw['license'] = $license;

		Settings::update_options($raw);
	}
}
