<?php

declare(strict_types=1);

namespace ContentReady\Generator;

use ContentReady\Modules\Registry;
use ContentReady\Toc\TocBuilder;

final class LocalGenerator
{
	private array $options;

	public function __construct(array $options)
	{
		$this->options = $options;
	}

	public function get_target_module_ids(array $meta): array
	{
		$page_type = is_string($meta['page_type'] ?? null) ? (string)$meta['page_type'] : 'article';
		$page_types = is_array($this->options['page_types'] ?? null) ? $this->options['page_types'] : [];

		foreach ($page_types as $row) {
			if (!is_array($row)) {
				continue;
			}
			if (($row['id'] ?? null) === $page_type && is_array($row['modules'] ?? null)) {
				return array_values($row['modules']);
			}
		}

		return Registry::all_ids();
	}

	public function generate_module(int $post_id, string $module_id, array $meta): array
	{
		$post = get_post($post_id);
		if (!$post) {
			return [];
		}

		$title = (string)get_the_title($post);
		$excerpt = (string)get_the_excerpt($post);

		switch ($module_id) {
			case 'summary':
				$text = $excerpt !== '' ? $excerpt : wp_trim_words((string)wp_strip_all_tags((string)$post->post_content), 60, '…');
				return [
					'text' => $text,
				];
			case 'verdict':
				return [
					'text' => $excerpt !== '' ? $excerpt : ('这篇内容主要围绕「' . $title . '」给出核心结论与可执行要点。'),
				];
			case 'tldr':
				return [
					'items' => [
						'先读结论：围绕「' . $title . '」提炼关键点。',
						'再看细节：按模块补齐 FAQ、限制与参考资料。',
						'最后检查：确保更新时间与结构化输出一致。',
					],
				];
			case 'limitations':
				return [
					'items' => [
						'仅适用于当前页面范围内的信息，不替代专业建议。',
						'当信息具有强时效性时，请确保更新时间准确。',
						'若与站点现有 SEO 插件冲突，结构化输出将自动降级。',
					],
				];
			case 'faq':
				return [
					'items' => [
						[
							'q' => '这篇内容的核心结论是什么？',
							'a' => '请查看「一句话结论」与「快速要点（TL;DR）」。',
						],
						[
							'q' => '我需要做哪些检查再发布？',
							'a' => '建议补齐 FAQ、参考资料与更新时间，并确认模块状态为已生成。',
						],
						[
							'q' => '会影响我现有主题样式吗？',
							'a' => '默认仅提供轻量样式，也可在设置中关闭样式输出。',
						],
					],
				];
			case 'references':
				return [
					'items' => [],
				];
			case 'last_updated':
				$modified = (string)get_post_modified_time('c', false, $post);
				return [
					'date' => $modified,
				];
			case 'toc':
				$items = TocBuilder::build_items((string)$post->post_content, $meta);
				return [
					'items' => $items,
				];
			case 'table':
				return [
					'caption' => '',
					'columns' => [],
					'rows' => [],
					'notes' => '',
				];
			case 'slug_suggestions':
				return [
					'suggestions' => [],
				];
			default:
				return [];
		}
	}
}
