<?php

declare(strict_types=1);

namespace ContentReady\Generator;

use ContentReady\AI\Client;
use ContentReady\Modules\AdaptiveRules;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\Util\ContentAnalyzer;

final class AiGenerator
{
	private Client $client;
	private array $options;
	private string $last_error = '';

	public function __construct(array $options)
	{
		$this->options = $options;
		$ai = is_array($options['ai'] ?? null) ? $options['ai'] : [];
		$this->client = new Client(
			(string)($ai['base_url'] ?? ''),
			(string)($ai['api_key'] ?? ''),
			(string)($ai['model'] ?? '')
		);
	}

	public function is_configured(): bool
	{
		return $this->client->is_configured();
	}

	public function get_last_error(): string
	{
		return $this->last_error;
	}

	public function get_target_module_ids(array $meta): array
	{
		$page_type = is_string($meta['page_type'] ?? null) ? (string)$meta['page_type'] : 'article';
		$page_types = is_array($this->options['page_types'] ?? null) ? $this->options['page_types'] : [];

		foreach ($page_types as $row) {
			if (!is_array($row)) {
				continue;
			}
			if (($row['id'] ?? null) === $page_type && is_array($row['modules'] ?? null)) {
				return array_values($row['modules']);
			}
		}

		return Registry::all_ids();
	}

	public function generate_module(int $post_id, string $module_id, array $meta): array
	{
		$post = get_post($post_id);
		if (!$post) {
			return [];
		}

		$this->last_error = '';

		// 内容信号分析
		$signals = ContentAnalyzer::analyze((string)$post->post_content, (string)get_the_title($post));

		// 自适应规则前置检测（仅当开关开启时）
		$adaptive_enabled = !empty($this->options['generation']['adaptive_enabled']);
		if ($adaptive_enabled && AdaptiveRules::should_skip($module_id, $signals)) {
			return [
				'_skipped' => true,
				'_reason'  => AdaptiveRules::get_skip_reason($module_id, $signals),
			];
		}

		$output_mode = AdaptiveRules::get_output_mode($module_id, $signals);
		$budget = AdaptiveRules::get_budget($signals['word_count']);

		$scope = (string)($this->options['privacy']['send_scope'] ?? 'full');
		$context = $this->build_context($post, $scope, $output_mode, $budget);

		$ai = is_array($this->options['ai'] ?? null) ? $this->options['ai'] : [];
		$mode = sanitize_key((string)($ai['mode'] ?? 'standard'));
		if (!in_array($mode, ['quick', 'standard', 'high'], true)) {
			$mode = 'standard';
		}

		$modules_cfg = ModulesConfig::normalize_modules_config($this->options['modules'] ?? null);
		$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
		$module_prompt = sanitize_textarea_field((string)($cfg['prompt'] ?? ''));

		// summary：若站点设置为“仅在摘要为空时写入（且摘要有内容则不生成）”，则直接复用现有摘要，避免不必要的 AI 调用。
		if ($module_id === 'summary') {
			$excerpt_mode = sanitize_key((string)($cfg['summary_excerpt'] ?? 'off'));
			if ($excerpt_mode === 'compat') {
				$existing_excerpt = trim((string)get_the_excerpt($post));
				if ($existing_excerpt !== '') {
					return ['text' => $existing_excerpt];
				}
			}
		}

		$system = "你是 WordPress 内容编辑助手。你必须只输出 JSON，不要输出任何解释性文字。";
		$user = $this->build_prompt($module_id, $mode, $module_prompt, $context);

		$res = $this->client->chat_json($system, $user, 30);
		if (empty($res['ok'])) {
			$this->last_error = is_string($res['error'] ?? null) ? (string)$res['error'] : 'ai_request_failed';
			return [];
		}

		$json = is_array($res['json'] ?? null) ? $res['json'] : [];
		return $this->extract_module_payload($module_id, $json);
	}

	private function build_context(\WP_Post $post, string $scope, string $output_mode = 'normal', int $budget = 0): array
	{
		$title = (string)get_the_title($post);
		$excerpt = (string)get_the_excerpt($post);
		$content = (string)$post->post_content;

		$content_text = trim(wp_strip_all_tags($content));
		$excerpt_text = trim(wp_strip_all_tags($excerpt));

		$base = [
			'_output_mode' => $output_mode,
			'_budget'      => $budget,
		];

		if ($scope === 'title_summary') {
			return array_merge($base, [
				'title'   => $title,
				'summary' => $excerpt_text !== '' ? $excerpt_text : wp_trim_words($content_text, 120, '…'),
			]);
		}

		if ($scope === 'selected_modules') {
			return array_merge($base, [
				'title'   => $title,
				'summary' => $excerpt_text !== '' ? $excerpt_text : wp_trim_words($content_text, 60, '…'),
			]);
		}

		return array_merge($base, [
			'title'   => $title,
			'excerpt' => $excerpt_text,
			'content' => $content_text,
		]);
	}

	private function build_prompt(string $module_id, string $mode, string $module_prompt, array $context): string
	{
		$budget = (int)($context['_budget'] ?? 0);
		$output_mode = (string)($context['_output_mode'] ?? 'normal');

		// 移除内部元数据，不传给 AI
		unset($context['_budget'], $context['_output_mode']);

		$base_rules = [
			"输出必须是合法 JSON。",
			"字段必须严格匹配要求的 schema，不能额外输出字段。",
			"尽量使用简体中文。",
			"允许使用基础排版 HTML：p、br、ul、ol、li、a、strong、em、code、pre；禁止任何自定义属性与脚本。",
		];

		// 预算约束
		if ($budget > 0) {
			$base_rules[] = "本文较短，所有模块输出总预算约 {$budget} 字，请控制本模块输出长度。";
		}

		// 输出模式约束
		if ($output_mode === 'short') {
			$base_rules[] = "文章较短，请输出精简版本，减少条目数量。";
		}

		if ($mode === 'quick') {
			$base_rules[] = "尽量简洁，优先短句与要点。";
		} elseif ($mode === 'high') {
			$base_rules[] = "尽量更全面，信息密度更高但避免废话。";
		}

		$ctx_json = wp_json_encode($context, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
		if (!is_string($ctx_json)) {
			$ctx_json = '{}';
		}

		$tldr_n = $mode === 'quick' ? '3' : ($mode === 'high' ? '5-7' : '3-5');
		$lim_n = $mode === 'quick' ? '3' : ($mode === 'high' ? '5-7' : '3-5');
		$faq_n = $mode === 'quick' ? '3-4' : ($mode === 'high' ? '5-8' : '3-6');
		$sum_len = $mode === 'quick' ? '80-150' : ($mode === 'high' ? '150-260' : '120-200');

		$extra = trim($module_prompt);
		$extra_block = '';
		if ($extra !== '') {
			$extra_block = "\n\n补充要求（来自站点设置的「模块提示词」；必须仍然严格遵守 schema）：\n" . $extra;
		}

		switch ($module_id) {
			case 'summary':
				return implode("\n", $base_rules) . "\n\n生成一段文章摘要（用于摘要/Excerpt），长度约 " . $sum_len . " 字。\n要求：不包含标题编号，不要分点，不要出现“本文/本篇”自指语。\nschema:\n{ \"text\": \"...\" }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'verdict':
				return implode("\n", $base_rules) . "\n\n根据下列上下文生成一句话结论。\nschema:\n{ \"text\": \"...\" }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'tldr':
				return implode("\n", $base_rules) . "\n\n生成 " . $tldr_n . " 条快速要点。\nschema:\n{ \"items\": [\"...\", \"...\"] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'limitations':
				return implode("\n", $base_rules) . "\n\n生成 " . $lim_n . " 条适用范围与限制。\nschema:\n{ \"items\": [\"...\", \"...\"] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'faq':
				return implode("\n", $base_rules) . "\n\n生成 " . $faq_n . " 条 FAQ。\nschema:\n{ \"items\": [{\"q\": \"...\", \"a\": \"...\"}] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'references':
				return implode("\n", $base_rules) . "\n\n如上下文中出现可作为来源的链接或明确来源信息，请输出参考资料；否则输出空数组。\nschema:\n{ \"items\": [{\"title\": \"\", \"url\": \"\"}] }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'table':
				return implode("\n", $base_rules) . "\n\n判断该文章是否值得生成一个信息密度高的表格（便于读者快速对比/归纳）。\n- 如果不值得生成：返回空表格（columns=[], rows=[]），并在 notes 写明“本文信息密度不足以生成表格表单”。\n- 如果值得生成：输出 3-6 列、3-10 行，单元格尽量简短。\nschema:\n{ \"caption\": \"\", \"columns\": [\"\"], \"rows\": [[\"...\"]], \"notes\": \"\" }\n\ncontext:\n" . $ctx_json . $extra_block;
			case 'slug_suggestions':
				return implode("\n", $base_rules) . "\n\n为该文章推荐 5 个英文 slug（用于 WordPress post_name）。\n要求：\n- 全小写，仅允许 a-z、0-9、连字符 '-'\n- 不要输出中文或十六进制编码\n- 每个 slug 2-6 个词，长度 20-60\n- 避免无意义词（the/a/of 等）\n\nschema:\n{ \"suggestions\": [\"a-b-c\"] }\n\ncontext:\n" . $ctx_json . $extra_block;
			default:
				return implode("\n", $base_rules) . "\n\n对该模块输出空对象。\nschema:\n{ }\n\ncontext:\n" . $ctx_json . $extra_block;
		}
	}

	private function extract_module_payload(string $module_id, array $json): array
	{
		switch ($module_id) {
			case 'summary':
				return ['text' => (string)($json['text'] ?? '')];
			case 'verdict':
				return ['text' => (string)($json['text'] ?? '')];
			case 'tldr':
			case 'limitations':
				return ['items' => is_array($json['items'] ?? null) ? $json['items'] : []];
			case 'faq':
				return ['items' => is_array($json['items'] ?? null) ? $json['items'] : []];
			case 'references':
				return ['items' => is_array($json['items'] ?? null) ? $json['items'] : []];
			case 'table':
				return [
					'caption' => (string)($json['caption'] ?? ''),
					'columns' => is_array($json['columns'] ?? null) ? $json['columns'] : [],
					'rows' => is_array($json['rows'] ?? null) ? $json['rows'] : [],
					'notes' => (string)($json['notes'] ?? ''),
				];
			case 'slug_suggestions':
				return [
					'suggestions' => is_array($json['suggestions'] ?? null) ? $json['suggestions'] : [],
				];
			default:
				return [];
		}
	}
}
