<?php

declare(strict_types=1);

namespace ContentReady\Frontend;

final class TablePlacement
{
	/**
	 * 将表格模块插入到“更自然”的正文位置：
	 * - 优先：匹配最相关的小节标题，插在该标题后的第一段落后
	 * - 兜底：插在第 3 段落后（避免出现在文章开头）
	 * - 最后兜底：追加到正文末尾
	 */
	public static function inject(string $content, string $table_html, array $table_data): string
	{
		if ($content === '' || $table_html === '') {
			return $content;
		}

		// 防止同一次请求里多次过滤导致重复注入。
		if (strpos($content, 'cr-module--table') !== false) {
			return $content;
		}

		$keywords = self::extract_keywords($table_data);

		$insert_at = self::find_best_heading_insertion_point($content, $keywords);
		if ($insert_at === null) {
			$insert_at = self::find_nth_paragraph_end($content, 3);
		}

		if ($insert_at === null) {
			return $content . $table_html;
		}

		return substr($content, 0, $insert_at) . $table_html . substr($content, $insert_at);
	}

	/**
	 * @return string[]
	 */
	private static function extract_keywords(array $table_data): array
	{
		$keywords = [];

		$caption = sanitize_text_field((string)($table_data['caption'] ?? ''));
		if ($caption !== '') {
			$keywords[] = $caption;
		}

		$cols = is_array($table_data['columns'] ?? null) ? $table_data['columns'] : [];
		foreach ($cols as $c) {
			$c = sanitize_text_field((string)$c);
			if ($c !== '') {
				$keywords[] = $c;
			}
		}

		$seed = implode(' ', array_slice($keywords, 0, 8));
		foreach (['参数', '规格', '配置', '对比', '清单', '推荐', '排名', '价格', '总结'] as $hint) {
			if ($seed !== '' && strpos($seed, $hint) !== false) {
				$keywords[] = $hint;
			}
		}

		$keywords = array_values(array_unique(array_filter(array_map('strval', $keywords), static fn(string $s): bool => trim($s) !== '')));

		// 避免超长关键词拖慢匹配。
		$out = [];
		foreach ($keywords as $k) {
			$k = trim($k);
			if ($k === '') {
				continue;
			}
			if (strlen($k) > 40) {
				$k = substr($k, 0, 40);
			}
			$out[] = $k;
		}

		return array_values(array_unique($out));
	}

	/**
	 * @param string[] $keywords
	 * @return int|null
	 */
	private static function find_best_heading_insertion_point(string $content, array $keywords): ?int
	{
		if ($keywords === []) {
			return null;
		}

		$ok = preg_match_all('~<(h[2-6])\\b[^>]*>(.*?)</\\1>~is', $content, $m, PREG_OFFSET_CAPTURE);
		if ($ok === false || $ok === 0) {
			return null;
		}

		$best_score = 0;
		$best_end = null;

		$matches = $m[0] ?? [];
		$inners = $m[2] ?? [];

		foreach ($matches as $idx => $full) {
			if (!is_array($full) || !isset($full[0], $full[1])) {
				continue;
			}
			$full_html = (string)$full[0];
			$offset = (int)$full[1];
			$end = $offset + strlen($full_html);

			$inner = isset($inners[$idx][0]) ? (string)$inners[$idx][0] : '';
			$text = trim(wp_strip_all_tags($inner));
			if ($text === '' || self::is_excluded_heading($text)) {
				continue;
			}

			$score = 0;
			foreach ($keywords as $k) {
				if ($k === '') {
					continue;
				}
				if (stripos($text, $k) !== false) {
					$score += 2;
					continue;
				}
				// 轻量兜底：关键词太长时取前 6 个字符做模糊命中。
				if (strlen($k) > 12) {
					$short = substr($k, 0, 6);
					if ($short !== '' && stripos($text, $short) !== false) {
						$score += 1;
					}
				}
			}

			if ($score > $best_score) {
				$best_score = $score;
				$best_end = $end;
			}
		}

		if ($best_score <= 0 || $best_end === null) {
			return null;
		}

		// 插在该标题后的第一段落后；没有段落则直接插在标题后。
		$after = substr($content, $best_end);
		if ($after === '') {
			return $best_end;
		}
		if (preg_match('~<p\\b[^>]*>.*?</p>~is', $after, $pm, PREG_OFFSET_CAPTURE) === 1) {
			$p_offset = (int)$pm[0][1];
			$p_html = (string)$pm[0][0];
			return $best_end + $p_offset + strlen($p_html);
		}

		return $best_end;
	}

	private static function is_excluded_heading(string $text): bool
	{
		$t = trim($text);
		if ($t === '') {
			return true;
		}

		foreach (['常见问题', 'FAQ', '参考', '引用', '结论', '总结', 'TL;DR', '摘要'] as $kw) {
			if (stripos($t, $kw) !== false) {
				return true;
			}
		}

		return false;
	}

	private static function find_nth_paragraph_end(string $content, int $n): ?int
	{
		if ($n <= 0) {
			return null;
		}

		$pos = 0;
		$needle = '</p>';
		for ($i = 0; $i < $n; $i++) {
			$found = stripos($content, $needle, $pos);
			if ($found === false) {
				return null;
			}
			$pos = (int)$found + strlen($needle);
		}

		return $pos;
	}
}
