<?php

declare(strict_types=1);

namespace ContentReady\Frontend;

use ContentReady\Admin\Settings;
use ContentReady\Licensing\Capabilities;
use ContentReady\Meta\MetaStore;
use ContentReady\Util\Locale;

final class StructuredData
{
	public static function register(): void
	{
		add_action('wp_head', [self::class, 'output_jsonld'], 99);
	}

	/**
	 * 结构化输出预览（用于编辑器侧展示/排查）。
	 *
	 * 返回：
	 * - objects: JSON-LD 对象数组（即使 auto 被阻断，也会尽量生成供预览）
	 * - would_output: 当前设置 + compat 状态下，前端是否会实际输出
	 * - blocked_reason: 仅当 would_output=false 时给出阻断原因 code
	 */
	public static function preview(int $post_id): array
	{
		$post = get_post($post_id);
		if (!$post) {
			return [
				'post_id' => $post_id,
				'objects' => [],
				'would_output' => false,
				'blocked_reason' => 'post_not_found',
			];
		}

		$options = Settings::get_options();
		$structured = $options['structured_output'] ?? [];
		$enabled = !empty($structured['enabled']);
		$mode = (string)($structured['mode'] ?? 'auto'); // auto | force | off

		$meta = MetaStore::get($post_id);
		$page_type = (string)($meta['page_type'] ?? 'article');

		$compat = is_array($meta['compat'] ?? null) ? $meta['compat'] : [];
		$checked_at = (int)($compat['checked_at'] ?? 0);
		$conflict = !empty($compat['jsonld_conflict']);
		$last_error = sanitize_text_field((string)($compat['last_error'] ?? ''));

		$would_output = false;
		$blocked_reason = '';
		if (!$enabled || $mode === 'off') {
			$blocked_reason = 'structured_disabled';
		} elseif ($page_type !== 'article') {
			$blocked_reason = 'not_article_page_type';
		} elseif ($mode === 'auto' && $checked_at <= 0) {
			$blocked_reason = 'auto_not_checked_yet';
		} elseif ($mode === 'auto' && $last_error !== '') {
			$blocked_reason = 'auto_check_failed';
		} elseif ($mode === 'auto' && $conflict) {
			$blocked_reason = 'auto_conflict_detected';
		} else {
			$would_output = true;
		}

		$objects = [];
		if ($page_type === 'article') {
			$objects[] = self::build_article($post, $options, $meta);

			$faq_items = $meta['modules']['faq']['data']['items'] ?? null;
			if (is_array($faq_items) && $faq_items !== []) {
				$faq = self::build_faq($faq_items, get_permalink($post));
				if ($faq) {
					$objects[] = $faq;
				}
			}
		}

		return [
			'post_id' => $post_id,
			'objects' => $objects,
			'would_output' => $would_output,
			'blocked_reason' => $blocked_reason,
			'structured_output' => [
				'enabled' => $enabled,
				'mode' => $mode,
			],
			'page_type' => $page_type,
			'compat' => [
				'checked_at' => $checked_at,
				'jsonld_conflict' => $conflict,
				'last_error' => $last_error,
			],
		];
	}

	public static function output_jsonld(): void
	{
		if (!is_singular() || is_admin()) {
			return;
		}

		$options = Settings::get_options();
		$enabled = !empty($options['structured_output']['enabled']);
		$mode = (string)($options['structured_output']['mode'] ?? 'auto'); // auto | force | off
		if (!$enabled || $mode === 'off') {
			return;
		}

		$post_id = (int)get_queried_object_id();
		if ($post_id <= 0) {
			return;
		}

		$meta = MetaStore::get($post_id);
		if (($meta['page_type'] ?? 'article') !== 'article') {
			return;
		}

		$checked_at = (int)($meta['compat']['checked_at'] ?? 0);
		$conflict = !empty($meta['compat']['jsonld_conflict']);
		$last_error = (string)($meta['compat']['last_error'] ?? '');
		if ($mode === 'auto' && $conflict) {
			return;
		}
		if ($mode === 'auto' && $last_error !== '') {
			return;
		}
		// auto 模式下：自检前先不输出，避免短时间内与 SEO 插件/主题的 JSON-LD 重复。
		if ($mode === 'auto' && $checked_at <= 0) {
			return;
		}

		$post = get_post($post_id);
		if (!$post) {
			return;
		}

		$objects = [];
		$objects[] = self::build_article($post, $options, $meta);

		$faq_items = $meta['modules']['faq']['data']['items'] ?? null;
		if (is_array($faq_items) && $faq_items !== []) {
			$faq = self::build_faq($faq_items, get_permalink($post));
			if ($faq) {
				$objects[] = $faq;
			}
		}

		foreach ($objects as $obj) {
			$json = wp_json_encode($obj, JSON_UNESCAPED_UNICODE);
			if (!is_string($json)) {
				continue;
			}
			echo '<script type="application/ld+json" data-cr-jsonld="1">' . $json . '</script>' . "\n";
		}
	}

	private static function build_article(\WP_Post $post, array $options, array $meta): array
	{
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$publisher_name = trim((string)($site['display_name'] ?? ''));
		if ($publisher_name === '') {
			$publisher_name = (string)get_bloginfo('name');
		}

		$is_pro = Capabilities::is_pro($options);
		$author_mode = 'wp';
		$brand_author_name = '';
		if ($is_pro) {
			$site_author_mode = (string)($site['author_mode'] ?? 'wp');
			if (!in_array($site_author_mode, ['wp', 'brand'], true)) {
				$site_author_mode = 'wp';
			}

			$override = is_array($meta['author'] ?? null) ? $meta['author'] : [];
			$override_mode = sanitize_key((string)($override['mode'] ?? 'inherit'));
			if (!in_array($override_mode, ['inherit', 'wp', 'brand'], true)) {
				$override_mode = 'inherit';
			}

			$author_mode = $override_mode === 'inherit' ? $site_author_mode : $override_mode;

			$brand_author_name = trim((string)($override['name'] ?? ''));
			if ($brand_author_name === '') {
				$brand_author_name = trim((string)($site['brand_author_name'] ?? ''));
			}
			if ($brand_author_name === '') {
				$brand_author_name = $publisher_name;
			}
		}

		$author_obj = null;
		if ($author_mode === 'brand') {
			$author_obj = [
				'@type' => 'Organization',
				'name' => $brand_author_name,
			];
		} else {
			$author_name = '';
			$author = get_user_by('id', (int)$post->post_author);
			if ($author && isset($author->display_name)) {
				$author_name = (string)$author->display_name;
			}
			if ($author_name !== '') {
				$author_obj = [
					'@type' => 'Person',
					'name' => $author_name,
				];
			}
		}

		$obj = [
			'@context' => 'https://schema.org',
			'@type' => 'Article',
			'mainEntityOfPage' => [
				'@type' => 'WebPage',
				'@id' => get_permalink($post),
			],
			'headline' => get_the_title($post),
			'datePublished' => get_the_date('c', $post),
			'dateModified' => get_post_modified_time('c', false, $post),
			'publisher' => [
				'@type' => 'Organization',
				'name' => $publisher_name,
			],
		];

		$publisher_url = home_url('/');
		if (is_string($publisher_url) && $publisher_url !== '') {
			$obj['publisher']['url'] = $publisher_url;
		}

		$logo = self::resolve_publisher_logo_url($site);
		if ($logo !== '') {
			$obj['publisher']['logo'] = [
				'@type' => 'ImageObject',
				'url' => $logo,
			];
		}

		$locale = Locale::normalize_bcp47((string)($site['locale'] ?? ''));
		if ($locale === '') {
			$locale = Locale::wp_locale_to_bcp47((string)get_locale());
		}
		if ($locale !== '') {
			$obj['inLanguage'] = $locale;
		}

		if (is_array($author_obj)) {
			$obj['author'] = $author_obj;
		}

		return $obj;
	}

	private static function resolve_publisher_logo_url(array $site): string
	{
		$explicit = esc_url_raw((string)($site['logo_url'] ?? ''));
		if ($explicit !== '') {
			return $explicit;
		}

		$icon = get_site_icon_url();
		if (is_string($icon) && $icon !== '') {
			return $icon;
		}

		$path = rtrim((string)ABSPATH, "/\\") . DIRECTORY_SEPARATOR . 'favicon.ico';
		if (is_file($path)) {
			return home_url('/favicon.ico');
		}

		$doc_root = isset($_SERVER['DOCUMENT_ROOT']) ? (string)$_SERVER['DOCUMENT_ROOT'] : '';
		if ($doc_root !== '') {
			$path2 = rtrim($doc_root, "/\\") . DIRECTORY_SEPARATOR . 'favicon.ico';
			if (is_file($path2)) {
				return home_url('/favicon.ico');
			}
		}

		return '';
	}

	private static function build_faq(array $items, string $permalink): ?array
	{
		$main = [];
		foreach ($items as $it) {
			if (!is_array($it)) {
				continue;
			}
			$q = sanitize_text_field((string)($it['q'] ?? ''));
			$a_html = (string)($it['a'] ?? '');
			$a = trim(wp_strip_all_tags($a_html));
			if ($q === '' || $a === '') {
				continue;
			}
			$main[] = [
				'@type' => 'Question',
				'name' => $q,
				'acceptedAnswer' => [
					'@type' => 'Answer',
					'text' => $a,
				],
			];
		}

		if ($main === []) {
			return null;
		}

		return [
			'@context' => 'https://schema.org',
			'@type' => 'FAQPage',
			'@id' => $permalink . '#faq',
			'mainEntity' => $main,
		];
	}
}
