<?php

declare(strict_types=1);

namespace ContentReady\Frontend;

use ContentReady\Admin\Settings;

final class ModuleRenderer
{
	public static function render(string $module_id, string $label, array $data, string $anchor): string
	{
		$label = $label !== '' ? $label : $module_id;

		switch ($module_id) {
			case 'summary':
				$text = sanitize_text_field((string)($data['text'] ?? ''));
				if ($text === '') {
					return '';
				}
				return self::wrap($module_id, $anchor, $label, '<div class="cr-module__body">' . esc_html($text) . '</div>');

			case 'verdict':
				$text = (string)($data['text'] ?? '');
				if ($text === '') {
					return '';
				}
				return self::wrap($module_id, $anchor, $label, '<div class="cr-module__body">' . $text . '</div>');

			case 'toc':
				$items_in = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items_in === []) {
					return '';
				}

				$options = Settings::get_options();
				$modules_cfg = is_array($options['modules'] ?? null) ? $options['modules'] : [];
				$cfg = is_array($modules_cfg['toc'] ?? null) ? $modules_cfg['toc'] : [];
				$rules = is_array($cfg['toc'] ?? null) ? $cfg['toc'] : [];

				$open = !empty($rules['default_open']);
				$list_style = sanitize_key((string)($rules['list_style'] ?? 'unordered'));
				if (!in_array($list_style, ['unordered', 'ordered'], true)) {
					$list_style = 'unordered';
				}
				$max_level = (int)($rules['max_level'] ?? 4);
				$max_level = max(2, min(5, $max_level));

				$items = [];
				foreach ($items_in as $it) {
					if (!is_array($it)) {
						continue;
					}
					$id = sanitize_text_field((string)($it['id'] ?? ''));
					$text = sanitize_text_field((string)($it['text'] ?? ''));
					$level = (int)($it['level'] ?? 2);
					$level = max(2, min(5, $level));
					if ($level > $max_level) {
						continue;
					}
					if ($id === '' || $text === '') {
						continue;
					}
					$items[] = ['id' => $id, 'text' => $text, 'level' => $level];
					if (count($items) >= 200) {
						break;
					}
				}

				// 少于 2 个标题时不展示目录（避免噪音）。
				if (count($items) < 2) {
					return '';
				}

				$tag = $list_style === 'ordered' ? 'ol' : 'ul';
				$list_class = 'cr-toc__list cr-toc__list--' . esc_attr($tag);
				$sub_class = 'cr-toc__sublist cr-toc__sublist--' . esc_attr($tag);

				$body = '<' . $tag . ' class="' . $list_class . '">';
				$current_level = 2;
				$first = true;

				foreach ($items as $it) {
					$level = (int)$it['level'];
					// 防止跳级导致非法嵌套：最多只下降/上升 1 级。
					if ($first) {
						if ($level > 2) {
							$level = 2;
						}
						$first = false;
					} elseif ($level > $current_level + 1) {
						$level = $current_level + 1;
					}

					if ($level === $current_level) {
						$body .= '</li>';
					} elseif ($level > $current_level) {
						$body .= '<' . $tag . ' class="' . $sub_class . '">';
						$current_level++;
					} else {
						$body .= '</li>';
						while ($current_level > $level) {
							$body .= '</' . $tag . '></li>';
							$current_level--;
						}
					}

					$body .= '<li class="cr-toc__item cr-toc__item--l' . esc_attr((string)$level) . '"><a href="#' . esc_attr((string)$it['id']) . '">' . esc_html((string)$it['text']) . '</a>';
				}

				$body .= '</li>';
				while ($current_level > 2) {
					$body .= '</' . $tag . '></li>';
					$current_level--;
				}
				$body .= '</' . $tag . '>';

				$open_attr = $open ? ' open' : '';
				$attr = ' class="cr-module cr-module--toc"';
				if ($anchor !== '') {
					$attr .= ' id="' . esc_attr($anchor) . '"';
				}

				return '<section' . $attr . '>' .
					'<details class="cr-toc"' . $open_attr . '>' .
					'<summary class="cr-toc__summary">' . esc_html($label) . '</summary>' .
					'<div class="cr-toc__body">' . $body . '</div>' .
					'</details>' .
					'</section>';

			case 'tldr':
			case 'limitations':
				$items = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items === []) {
					return '';
				}
				$list = '<ul class="cr-module__list">';
				foreach ($items as $item) {
					$list .= '<li>' . (string)$item . '</li>';
				}
				$list .= '</ul>';
				return self::wrap($module_id, $anchor, $label, $list);

			case 'faq':
				$items = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items === []) {
					return '';
				}
				$out = '<div class="cr-faq">';
				foreach ($items as $item) {
					if (!is_array($item)) {
						continue;
					}
					$q = isset($item['q']) ? esc_html((string)$item['q']) : '';
					$a = isset($item['a']) ? (string)$item['a'] : '';
					if ($q === '' || $a === '') {
						continue;
					}
					$out .= '<div class="cr-faq__item">';
					$out .= '<div class="cr-faq__q"><strong>Q：</strong>' . $q . '</div>';
					$out .= '<div class="cr-faq__a"><strong>A：</strong>' . $a . '</div>';
					$out .= '</div>';
				}
				$out .= '</div>';
				return self::wrap($module_id, $anchor, $label, $out);

			case 'references':
				$items = is_array($data['items'] ?? null) ? $data['items'] : [];
				if ($items === []) {
					return '';
				}
				$out = '<ul class="cr-module__list">';
				foreach ($items as $item) {
					if (!is_array($item)) {
						continue;
					}
					$title = esc_html((string)($item['title'] ?? ''));
					$url = esc_url((string)($item['url'] ?? ''));
					if ($url === '') {
						continue;
					}
					$text = $title !== '' ? $title : $url;
					$out .= '<li><a href="' . $url . '" rel="nofollow noopener" target="_blank">' . $text . '</a></li>';
				}
				$out .= '</ul>';
				return self::wrap($module_id, $anchor, $label, $out);

			case 'last_updated':
				$date = sanitize_text_field((string)($data['date'] ?? ''));
				if ($date === '') {
					return '';
				}
				$ts = strtotime($date);
				if (!$ts) {
					return self::wrap($module_id, $anchor, $label, '<div class="cr-module__body">' . esc_html($date) . '</div>');
				}

				$options = Settings::get_options();
				$site = is_array($options['site'] ?? null) ? $options['site'] : [];
				$fmt = (string)($site['date_format'] ?? 'wp');
				if ($fmt === '' || $fmt === 'wp') {
					$fmt = (string)get_option('date_format');
				}
				$human = wp_date($fmt, $ts);
				$datetime = wp_date('c', $ts);
				$body = '<div class="cr-module__body"><time class="cr-module__time" datetime="' . esc_attr($datetime) . '">' . esc_html($human) . '</time></div>';
				return self::wrap($module_id, $anchor, $label, $body);

			case 'table':
				$caption = sanitize_text_field((string)($data['caption'] ?? ''));
				$notes = sanitize_text_field((string)($data['notes'] ?? ''));
				$cols = is_array($data['columns'] ?? null) ? $data['columns'] : [];
				$rows = is_array($data['rows'] ?? null) ? $data['rows'] : [];

				$cols = array_values(array_filter(array_map('sanitize_text_field', array_map('strval', $cols)), static fn($s) => $s !== ''));
				$clean_rows = [];
				foreach ($rows as $row) {
					if (!is_array($row)) {
						continue;
					}
					$r = array_values(array_map('sanitize_text_field', array_map('strval', $row)));
					$r = array_map(static fn($s) => trim((string)$s), $r);
					if ($r !== []) {
						$clean_rows[] = $r;
					}
				}
				$rows = $clean_rows;

				if ($cols === [] || $rows === []) {
					$msg = $notes !== '' ? $notes : '本文信息密度不足以生成表格表单';
					return self::wrap($module_id, $anchor, $label, '<div class="cr-module__body cr-module__muted">' . esc_html($msg) . '</div>');
				}

				$out = '<div class="cr-module__body">';
				$out .= '<div class="cr-table-wrap"><table class="cr-table">';
				if ($caption !== '') {
					$out .= '<caption>' . esc_html($caption) . '</caption>';
				}
				$out .= '<thead><tr>';
				foreach ($cols as $c) {
					$out .= '<th scope="col">' . esc_html((string)$c) . '</th>';
				}
				$out .= '</tr></thead><tbody>';
				foreach ($rows as $row) {
					if (!is_array($row)) {
						continue;
					}
					$out .= '<tr>';
					foreach ($cols as $idx => $_c) {
						$cell = isset($row[$idx]) ? (string)$row[$idx] : '';
						$out .= '<td>' . esc_html($cell) . '</td>';
					}
					$out .= '</tr>';
				}
				$out .= '</tbody></table></div>';
				if ($notes !== '') {
					$out .= '<p class="cr-table__notes">' . esc_html($notes) . '</p>';
				}
				$out .= '</div>';

				return self::wrap($module_id, $anchor, $label, $out);

			case 'slug_suggestions':
				// 仅编辑器辅助功能：不在前端渲染。
				return '';

			default:
				return '';
		}
	}

	private static function wrap(string $module_id, string $anchor, string $label, string $body_html): string
	{
		$attr = ' class="cr-module cr-module--' . esc_attr($module_id) . '"';
		if ($anchor !== '') {
			$attr .= ' id="' . esc_attr($anchor) . '"';
		}

		return '<section' . $attr . '>' .
			'<h2 class="cr-module__title">' . esc_html($label) . '</h2>' .
			$body_html .
			'</section>';
	}
}
