<?php

declare(strict_types=1);

namespace ContentReady\Editor;

use ContentReady\Admin\Settings;

final class Classic
{
	public static function register(): void
	{
		add_action('add_meta_boxes', [self::class, 'add_metabox']);
		add_action('admin_enqueue_scripts', [self::class, 'enqueue_assets']);
	}

	public static function enqueue_assets(string $hook): void
	{
		if (!in_array($hook, ['post.php', 'post-new.php'], true)) {
			return;
		}

		$screen = function_exists('get_current_screen') ? get_current_screen() : null;
		if (!$screen || empty($screen->base)) {
			return;
		}

		$post_id = isset($_GET['post']) ? (int)$_GET['post'] : 0;
		if ($post_id <= 0) {
			global $post;
			if ($post instanceof \WP_Post) {
				$post_id = (int)$post->ID;
			}
		}
		if ($post_id <= 0) {
			return;
		}

		if (!current_user_can('edit_post', $post_id)) {
			return;
		}

		wp_enqueue_script(
			'content-ready-pro-upsell',
			CR_PLUGIN_URL . 'assets/pro-upsell.js',
			[],
			(string)(filemtime(CR_PLUGIN_DIR . 'assets/pro-upsell.js') ?: CR_VERSION),
			true
		);

		wp_enqueue_style(
			'content-ready-pro-upsell',
			CR_PLUGIN_URL . 'assets/pro-upsell.css',
			[],
			(string)(filemtime(CR_PLUGIN_DIR . 'assets/pro-upsell.css') ?: CR_VERSION)
		);

		wp_enqueue_script(
			'content-ready-classic-utils',
			CR_PLUGIN_URL . 'assets/classic-utils.js',
			[],
			CR_VERSION,
			true
		);

		wp_enqueue_script(
			'content-ready-payload-utils',
			CR_PLUGIN_URL . 'assets/payload-utils.js',
			[],
			CR_VERSION,
			true
		);

		wp_enqueue_script(
			'content-ready-classic-slug',
			CR_PLUGIN_URL . 'assets/classic-slug.js',
			['content-ready-pro-upsell'],
			CR_VERSION,
			true
		);

		wp_enqueue_script(
			'content-ready-classic-checks',
			CR_PLUGIN_URL . 'assets/classic-checks.js',
			[],
			CR_VERSION,
			true
		);

		wp_enqueue_script(
			'content-ready-classic',
			CR_PLUGIN_URL . 'assets/classic.js',
			['content-ready-pro-upsell', 'content-ready-classic-utils', 'content-ready-payload-utils', 'content-ready-classic-slug', 'content-ready-classic-checks', 'wp-api-fetch'],
			CR_VERSION,
			true
		);

		wp_localize_script('content-ready-classic', 'CR_CLASSIC', [
			'postId' => $post_id,
			'restNonce' => wp_create_nonce('wp_rest'),
			'settingsUrl' => admin_url('admin.php?page=content-ready-settings'),
			'licenseUrl' => admin_url('admin.php?page=content-ready-settings&tab=license'),
			'proBuyUrl' => apply_filters('cr_pro_buy_url', 'https://xmtbx.com/content-ready'),
		]);

		wp_enqueue_style(
			'content-ready-badges',
			CR_PLUGIN_URL . 'assets/badges.css',
			[],
			CR_VERSION
		);

		wp_enqueue_script(
			'content-ready-classic-structured-preview',
			CR_PLUGIN_URL . 'assets/classic-structured-preview.js',
			['content-ready-classic', 'wp-api-fetch'],
			CR_VERSION,
			true
		);
	}

	public static function add_metabox(): void
	{
		$options = Settings::get_options();
		$ui = is_array($options['ui'] ?? null) ? $options['ui'] : [];
		$title = trim(sanitize_text_field((string)($ui['editor_entry_label'] ?? '')));
		if ($title === '') {
			$title = '内容就绪';
		}

		add_meta_box(
			'content-ready-metabox',
			esc_html($title),
			[self::class, 'render_metabox'],
			null,
			'side',
			'default'
		);
	}

	public static function render_metabox(\WP_Post $post): void
	{
		if (!current_user_can('edit_post', $post->ID)) {
			return;
		}

		echo '<div id="cr-classic-root" data-post-id="' . esc_attr((string)$post->ID) . '">';
		echo '<p>' . esc_html__('加载中…（如无响应，请确认 REST 已启用且无安全插件拦截）', 'content-ready') . '</p>';
		echo '</div>';
	}
}
