<?php

declare(strict_types=1);

namespace ContentReady\Checks;

use ContentReady\Admin\Settings;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\Generator\V1Generator;

final class Checker
{
	/**
	 * 返回：
	 * - summary: { pass, warn, fail }
	 * - items: [{ level, code, message, module_id? }]
	 */
	public static function check_post(int $post_id): array
	{
		$options = Settings::get_options();
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);

		$meta = MetaStore::get($post_id);
		$generator = new V1Generator($options);
		$target_ids = $generator->get_target_module_ids($meta);

		$items = [];

		foreach ($target_ids as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}
			if ($module_id === 'slug_suggestions') {
				// 仅编辑器辅助功能：不参与发布前检查。
				continue;
			}

			$data = $meta['modules'][$module_id]['data'] ?? null;
			$missing = !is_array($data) || $data === [];
			if ($missing) {
				$items[] = [
					'level' => 'fail',
					'code' => 'missing_module',
					'module_id' => $module_id,
					'message' => '模块缺失：' . (string)($cfg['label'] ?? $module_id),
				];
				continue;
			}

			if ($module_id === 'faq') {
				$min = (int)($cfg['min_rules']['faq_min_items'] ?? 3);
				$list = is_array($data['items'] ?? null) ? $data['items'] : [];
				if (count($list) > 0 && count($list) < $min) {
					$items[] = [
						'level' => 'warn',
						'code' => 'faq_too_few',
						'module_id' => $module_id,
						'message' => 'FAQ 条数不足：当前 ' . count($list) . '，最少 ' . $min,
					];
				}
			}
		}

		$structured = $options['structured_output'] ?? [];
		$mode = (string)($structured['mode'] ?? 'auto');
		$so_enabled = !empty($structured['enabled']) && ($mode !== 'off');
		$conflict = !empty($meta['compat']['jsonld_conflict']);
		$checked_at = (int)($meta['compat']['checked_at'] ?? 0);
		$last_error = sanitize_text_field((string)($meta['compat']['last_error'] ?? ''));

		if ($so_enabled && ($mode === 'auto') && $checked_at <= 0) {
			$items[] = [
				'level' => 'warn',
				'code' => 'jsonld_conflict_not_checked_yet',
				'message' => '结构化输出处于“智能兼容”：正在检查是否与其他结构化重复；完成前暂不输出（可稍后再看前台；如需立即输出可切换为“强制输出”（Pro））。',
			];
		}
		if ($so_enabled && ($mode === 'auto') && $checked_at > 0 && $last_error !== '') {
			$items[] = [
				'level' => 'warn',
				'code' => 'jsonld_conflict_check_failed',
				'message' => '结构化输出兼容性检查失败：' . $last_error . '（“智能兼容”下为安全起见将暂停输出；你可检查站点是否阻止回源抓取（loopback/WAF），或切换为“强制输出”（Pro））。',
			];
		}
		if ($so_enabled && $conflict && ($mode === 'auto')) {
			$items[] = [
				'level' => 'warn',
				'code' => 'jsonld_conflict_auto_disabled',
				'message' => '检测到站点已有其他结构化输出（可能来自 SEO 插件/主题）：为避免重复，“智能兼容”已暂停本插件结构化输出。若希望由本插件输出：请在对应 SEO 插件中关闭结构化输出（Schema/JSON-LD）（全局/该文章类型/单篇文章），或切换为“强制输出”（Pro）。',
			];
		}

		$summary = ['pass' => 0, 'warn' => 0, 'fail' => 0];
		foreach ($items as $it) {
			$level = (string)($it['level'] ?? '');
			if (isset($summary[$level])) {
				$summary[$level]++;
			}
		}

		if ($summary['fail'] === 0 && $summary['warn'] === 0) {
			$items[] = [
				'level' => 'pass',
				'code' => 'ok',
				'message' => '检查通过：未发现问题。',
			];
			$summary['pass'] = 1;
		}

		return [
			'summary' => $summary,
			'items' => $items,
		];
	}

	/**
	 * 用于持久化到 post_meta 的检查结果：限制字段与长度，避免写入过大或不稳定的数据。
	 */
	public static function sanitize_result($result): array
	{
		$result = is_array($result) ? $result : [];
		$summary_in = is_array($result['summary'] ?? null) ? $result['summary'] : [];
		$items_in = is_array($result['items'] ?? null) ? $result['items'] : [];

		$summary = [
			'pass' => (int)($summary_in['pass'] ?? 0),
			'warn' => (int)($summary_in['warn'] ?? 0),
			'fail' => (int)($summary_in['fail'] ?? 0),
		];

		$items = [];
		foreach ($items_in as $it) {
			if (!is_array($it)) {
				continue;
			}
			$level = sanitize_key((string)($it['level'] ?? ''));
			if (!in_array($level, ['pass', 'warn', 'fail'], true)) {
				$level = 'warn';
			}
			$code = sanitize_key((string)($it['code'] ?? ''));
			$message = sanitize_text_field((string)($it['message'] ?? $code));
			$module_id = sanitize_key((string)($it['module_id'] ?? ''));

			$row = [
				'level' => $level,
				'code' => $code,
				'message' => $message,
			];
			if ($module_id !== '') {
				$row['module_id'] = $module_id;
			}
			$items[] = $row;
			if (count($items) >= 50) {
				break;
			}
		}

		return [
			'summary' => $summary,
			'items' => $items,
		];
	}
}

