<?php

declare(strict_types=1);

namespace ContentReady\Automation;

use ContentReady\Admin\Settings;
use ContentReady\Generator\V1Generator;
use ContentReady\Licensing\Capabilities;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\Modules\Sanitizer;

final class GeoRunner
{
	/**
	 * 为指定文章执行一次“自动 GEO”。
	 *
	 * @return array{generated: string[], skipped: string[], failed: array<string,string>}
	 */
	public static function run_for_post(int $post_id, bool $only_missing = true, bool $skip_edited = true): array
	{
		$post = get_post($post_id);
		if (!$post || $post->post_status !== 'publish') {
			return ['generated' => [], 'skipped' => [], 'failed' => []];
		}

		$options = Settings::get_options();
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$generator = new V1Generator($options);
		$meta = MetaStore::get($post_id);

		$generated = [];
		$skipped = [];
		$failed = [];

		foreach ($generator->get_target_module_ids($meta) as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			if (!Capabilities::is_module_available($module_id, $options)) {
				continue;
			}
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}

			$m = is_array($meta['modules'][$module_id] ?? null) ? $meta['modules'][$module_id] : [];
			$gen_at = (int)($m['generated_at'] ?? 0);
			$edit_at = (int)($m['edited_at'] ?? 0);
			if ($skip_edited && $edit_at > 0 && $gen_at > 0 && $edit_at > $gen_at) {
				$skipped[] = $module_id;
				continue;
			}

			$data = $m['data'] ?? null;
			if ($only_missing && is_array($data) && $data !== []) {
				continue;
			}

			if ($generator->requires_ai($module_id) && !$generator->is_ai_configured()) {
				$failed[$module_id] = 'ai_not_configured';
				continue;
			}

			$out = $generator->generate_module($post_id, $module_id, $meta);
			if (!empty($out['_skipped'])) {
				$skipped[] = $module_id;
				continue;
			}
			if ($out === []) {
				$failed[$module_id] = $generator->get_ai_last_error();
				continue;
			}

			$clean = Sanitizer::sanitize_module_data($module_id, $out);
			$meta = MetaStore::touch_generated($post_id, $module_id, $clean);
			self::maybe_write_summary_excerpt($post_id, $modules_cfg, $module_id, $clean);
			$generated[] = $module_id;
		}

		return [
			'generated' => $generated,
			'skipped' => $skipped,
			'failed' => $failed,
		];
	}

	private static function maybe_write_summary_excerpt(int $post_id, array $modules_cfg, string $module_id, array $data): void
	{
		if ($module_id !== 'summary') {
			return;
		}

		$cfg = is_array($modules_cfg['summary'] ?? null) ? $modules_cfg['summary'] : [];
		$mode = sanitize_key((string)($cfg['summary_excerpt'] ?? 'off'));
		if (!in_array($mode, ['off', 'force', 'compat'], true)) {
			$mode = 'off';
		}
		if ($mode === 'off') {
			return;
		}

		$text = sanitize_text_field((string)($data['text'] ?? ''));
		$text = trim((string)preg_replace('/\\s+/u', ' ', $text));
		if ($text === '') {
			return;
		}

		$post = get_post($post_id);
		if (!$post) {
			return;
		}

		$current = trim((string)$post->post_excerpt);
		if ($mode === 'compat' && $current !== '') {
			return;
		}

		if ($current === $text) {
			return;
		}

		wp_update_post([
			'ID' => $post_id,
			'post_excerpt' => $text,
		]);
	}
}
