<?php

declare(strict_types=1);

namespace ContentReady\Admin\Ui;

use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Registry;

final class PlanCompare
{
	public static function render(array $options): void
	{
		$plan = Capabilities::plan($options);

		echo '<hr />';
		echo '<h3 style="margin-top:16px;">免费版 vs Pro 功能对比</h3>';
		echo '<p class="description">该表格由能力矩阵自动渲染：新增模块/功能后，无需改页面代码即可展示。</p>';
		echo '<p class="description">当前计划：' . esc_html($plan === Capabilities::PLAN_PRO ? 'Pro' : '免费版') . '</p>';

		echo '<table class="widefat striped cr-plan-compare" style="margin-top:12px;">';
		echo '<thead><tr>';
		echo '<th>功能</th>';
		echo '<th class="cr-plan-compare__center">免费版</th>';
		echo '<th class="cr-plan-compare__center cr-plan-compare__pro">Pro</th>';
		echo '</tr></thead>';
		echo '<tbody>';

		self::render_features();
		self::render_modules();

		echo '</tbody></table>';
	}

	private static function render_features(): void
	{
		$defs = Capabilities::feature_definitions();
		if ($defs === []) {
			return;
		}

		$groups = [];
		foreach ($defs as $key => $def) {
			$group = (string)($def['group'] ?? '其他');
			if (!isset($groups[$group])) {
				$groups[$group] = [];
			}
			$groups[$group][$key] = $def;
		}

		foreach ($groups as $group => $items) {
			// 同组内：先展示 Free+Pro 都可用的功能，再展示 Pro-only（更符合阅读预期）。
			uasort($items, static function ($a, $b): int {
				$pa = !empty($a['pro_only']);
				$pb = !empty($b['pro_only']);
				if ($pa === $pb) {
					return 0;
				}
				return $pa ? 1 : -1;
			});

			echo '<tr class="cr-plan-compare__group"><th colspan="3">' . esc_html($group) . '</th></tr>';
			foreach ($items as $def) {
				$label = (string)($def['label'] ?? '');
				$desc = (string)($def['description'] ?? '');
				$pro_only = !empty($def['pro_only']);

				echo '<tr>';
				echo '<td><strong>' . esc_html($label) . '</strong>' . ($desc !== '' ? ('<div class="description" style="margin:4px 0 0;">' . esc_html($desc) . '</div>') : '') . '</td>';
				echo '<td class="cr-plan-compare__center">' . self::icon(!$pro_only) . '</td>';
				echo '<td class="cr-plan-compare__center cr-plan-compare__pro">' . self::icon(true) . '</td>';
				echo '</tr>';
			}
		}
	}

	private static function render_modules(): void
	{
		$all = Registry::all();

		$content_modules = [];
		foreach (Registry::all_ids() as $module_id) {
			if (!Registry::is_content_module($module_id)) {
				continue;
			}
			$content_modules[$module_id] = $all[$module_id] ?? [];
		}
		if ($content_modules === []) {
			return;
		}

		// 先展示 Free 可用模块，再展示 Pro 模块（更符合阅读预期）。
		$free_first = [];
		$pro_later = [];
		foreach ($content_modules as $module_id => $cfg) {
			if (Capabilities::module_plan($module_id) === Capabilities::PLAN_FREE) {
				$free_first[$module_id] = $cfg;
			} else {
				$pro_later[$module_id] = $cfg;
			}
		}
		$content_modules = $free_first + $pro_later;

		echo '<tr class="cr-plan-compare__group"><th colspan="3">内容模块库</th></tr>';
		foreach ($content_modules as $module_id => $cfg) {
			$label = (string)($cfg['label'] ?? $module_id);
			$free = Capabilities::module_plan($module_id) === Capabilities::PLAN_FREE;

			echo '<tr>';
			echo '<td><strong>' . esc_html($label) . '</strong></td>';
			echo '<td class="cr-plan-compare__center">' . self::icon($free) . '</td>';
			echo '<td class="cr-plan-compare__center cr-plan-compare__pro">' . self::icon(true) . '</td>';
			echo '</tr>';
		}
	}

	private static function icon(bool $enabled): string
	{
		if ($enabled) {
			return '<span class="dashicons dashicons-yes-alt" aria-hidden="true"></span><span class="screen-reader-text">支持</span>';
		}
		return '<span class="dashicons dashicons-minus" aria-hidden="true"></span><span class="screen-reader-text">不支持</span>';
	}
}

