<?php

declare(strict_types=1);

namespace ContentReady\Admin\SettingsSections;

use ContentReady\Admin\Ui\Badges;
use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Registry;
use ContentReady\PageTypes\Config as PageTypesConfig;

final class PageTypesSection
{
	public static function render(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);
		$all_modules = Registry::all();

		echo '<hr />';
		echo '<h2>' . esc_html__('页面类型规则', 'content-ready') . ' <span class="cr-badge-wrap">' . Badges::pro('新增/删除页面类型为 Pro 功能') . '</span></h2>';
		echo '<p>' . esc_html__('页面类型用来描述“这篇内容属于哪一类”（例如：通用文章、教程、评测等）。插件会据此决定：生成哪些内容模块、以及对应的结构化输出策略（后续版本将逐步补齐）。', 'content-ready') . '</p>';
		echo '<p>' . esc_html__('页面类型只保存在文章的插件配置里，不会改写正文。建议：先从「通用文章」开始，跑通一套模块组合，再逐步拆分更细的类型。', 'content-ready') . '</p>';
		echo '<p>' . esc_html__('提示：标识（id）用于识别页面类型，创建后尽量不要修改；删除后，已有文章会自动按「通用文章」规则处理。', 'content-ready') . '</p>';

		$page_types = is_array($options['page_types'] ?? null) ? $options['page_types'] : [];
		$page_types = PageTypesConfig::normalize_page_types($page_types);
		$content_ids = array_values(array_filter(ModulesConfig::default_order(), [Registry::class, 'is_content_module']));
		$content_ids = array_values(array_filter($content_ids, static function (string $mid) use ($options): bool {
			return Capabilities::is_module_available($mid, $options);
		}));

		echo '<table class="widefat striped">';
		echo '<thead><tr>';
		echo '<th style="width:140px;">标识（id）</th>';
		echo '<th style="width:220px;">名称</th>';
		echo '<th>模块组合</th>';
		echo '<th style="width:120px;">删除</th>';
		echo '</tr></thead><tbody>';

		foreach ($page_types as $pt) {
			if (!is_array($pt)) {
				continue;
			}
			$id = (string)($pt['id'] ?? '');
			$label = (string)($pt['label'] ?? '');
			$mods = is_array($pt['modules'] ?? null) ? $pt['modules'] : [];
			$locked = !$is_pro && $id !== 'article';
			$disabled = $locked ? ' disabled' : '';
			$set = [];
			foreach ($mods as $m) {
				$set[(string)$m] = true;
			}

			echo '<tr>';
			echo '<td><code>' . esc_html($id) . '</code><input type="hidden" name="page_types[' . esc_attr($id) . '][id]" value="' . esc_attr($id) . '" /></td>';
			echo '<td><input class="regular-text" name="page_types[' . esc_attr($id) . '][label]" value="' . esc_attr($label) . '"' . $disabled . ' />' . ($id !== 'article' ? (' <span class="cr-badge-wrap">' . Badges::pro('该页面类型为 Pro（Free 仅可使用通用文章）') . '</span>') : '') . '</td>';

			echo '<td>';
			foreach ($content_ids as $mid) {
				$checked = isset($set[$mid]);
				$mlabel = (string)($all_modules[$mid]['label'] ?? $mid);
				echo '<label style="display:inline-block;margin-right:10px;"><input type="checkbox" name="page_types[' . esc_attr($id) . '][modules][' . esc_attr($mid) . ']" value="1"' . checked($checked, true, false) . $disabled . ' /> ' . esc_html($mlabel) . '</label>';
			}
			echo '</td>';

			echo '<td>';
			if ($id === 'article') {
				echo '<span style="color:#6b7280;">不可删除</span>';
			} elseif (!$is_pro) {
				echo '<span style="color:#6b7280;">Pro 可删除</span>';
			} else {
				echo '<label><input type="checkbox" name="page_types_delete[]" value="' . esc_attr($id) . '" /> 删除</label>';
			}
			echo '</td>';
			echo '</tr>';
		}

		echo '</tbody></table>';

		if ($is_pro) {
			echo '<h3>' . esc_html__('新增页面类型', 'content-ready') . '</h3>';
			echo '<table class="form-table" role="presentation"><tbody>';
			echo '<tr><th scope="row"><label for="cr_pt_new_id">标识（id）</label></th><td><input class="regular-text" id="cr_pt_new_id" name="page_type_new_id" placeholder="例如：howto / review（建议英文短词）" /></td></tr>';
			echo '<tr><th scope="row"><label for="cr_pt_new_label">名称</label></th><td><input class="regular-text" id="cr_pt_new_label" name="page_type_new_label" placeholder="例如：教程" /></td></tr>';
			echo '<tr><th scope="row">模块组合</th><td>';
			foreach ($content_ids as $mid) {
				$mlabel = (string)($all_modules[$mid]['label'] ?? $mid);
				echo '<label style="display:inline-block;margin-right:10px;"><input type="checkbox" name="page_type_new_modules[' . esc_attr($mid) . ']" value="1" /> ' . esc_html($mlabel) . '</label>';
			}
			echo '</td></tr>';
			echo '</tbody></table>';
		} else {
			echo '<p class="description">Free：仅支持使用「通用文章」并配置其免费模块组合；Pro 可新增/删除页面类型，并为不同类型配置不同的模块组合。</p>';
		}
	}
}
