<?php

declare(strict_types=1);

namespace ContentReady\Admin\SettingsSections;

use ContentReady\Admin\Ui\Badges;
use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Dispatch as ModulesDispatch;
use ContentReady\Modules\Registry;

final class ModulesSection
{
	public static function render(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);
		$modules_cfg = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$order = ModulesConfig::default_order();
		$dispatch_mode = sanitize_key((string)($options['generation']['dispatch_mode'] ?? ModulesDispatch::MODE_OFF));
		if (!in_array($dispatch_mode, [ModulesDispatch::MODE_OFF, ModulesDispatch::MODE_ADAPTIVE, ModulesDispatch::MODE_LIGHTWEIGHT], true)) {
			$dispatch_mode = !empty($options['generation']['adaptive_enabled']) ? ModulesDispatch::MODE_ADAPTIVE : ModulesDispatch::MODE_OFF;
		}
		$readonly = $is_pro && $dispatch_mode === ModulesDispatch::MODE_LIGHTWEIGHT;
		$all_modules = Registry::all();

		echo '<h3 style="margin-top:20px;">模块配置</h3>';

		// 模块调度策略（Pro-only）
		echo '<table class="form-table" role="presentation"><tr>';
		echo '<th scope="row">模块调度策略 <span class="cr-badge-wrap">' . Badges::pro('模块调度策略为 Pro 功能') . Badges::ai('将影响 AI 调用与 Token 消耗') . '</span></th>';
		echo '<td>';

		$mode_labels = [
			ModulesDispatch::MODE_OFF => '关闭（完全按模块开关生成）',
			ModulesDispatch::MODE_ADAPTIVE => '自适应模块调度',
			ModulesDispatch::MODE_LIGHTWEIGHT => '固定轻量化模块调度（推荐组合）',
		];

		if ($is_pro) {
			echo '<select name="generation_dispatch_mode">';
			foreach ($mode_labels as $k => $label) {
				echo '<option value="' . esc_attr($k) . '"' . selected($dispatch_mode, $k, false) . '>' . esc_html($label) . '</option>';
			}
			echo '</select>';

			echo '<p class="description">关闭：完全按你启用的模块生成。自适应：根据正文信号自动跳过不适合的模块，避免结构膨胀。固定轻量化：使用推荐组合减少 AI 调用次数，优先保证通用收益与成本可控。</p>';

			$lw = ModulesDispatch::lightweight_recommended_module_ids();
			$lw_labels = [];
			foreach ($lw as $mid) {
				$mid = sanitize_key((string)$mid);
				$lw_labels[] = (string)($all_modules[$mid]['label'] ?? $mid);
			}
			echo '<p class="description">轻量化推荐组合：' . esc_html(implode('、', $lw_labels)) . '。</p>';
			if ($readonly) {
				echo '<p class="description"><strong>提示：</strong>轻量化模式下模块配置为只读；如需自定义模块开关与提示词，请先将调度策略切换为“关闭”。</p>';
			}
		} else {
			echo '<select disabled>';
			foreach ($mode_labels as $k => $label) {
				echo '<option value="' . esc_attr($k) . '"' . selected($dispatch_mode, $k, false) . '>' . esc_html($label) . '</option>';
			}
			echo '</select>';
			echo '<p class="description">Free：该功能为 Pro。你仍可在下方启用已解锁的模块。</p>';
		}

		echo '</td></tr></table>';

		echo '<p class="description" style="margin-top:10px;">你可以配置：启用、显示名称、规则（如：FAQ 最少条数、摘要写入）以及提示词（可选，用于影响 AI 生成）。模块的顺序与展示位置由插件统一规划（不需要你手动调整）。</p>';

		echo '<div class="cr-modules-toolbar">';
		echo '<label class="cr-modules-search">';
		echo '<span class="screen-reader-text">搜索模块</span>';
		echo '<input type="search" id="cr-modules-search" placeholder="搜索模块（名称/简介/ID）" />';
		echo '</label>';
		echo '<span id="cr-modules-search-count" class="description"></span>';
		echo '<p id="cr-modules-search-empty" class="description cr-modules-search-empty">未找到匹配的模块。</p>';
		echo '</div>';

		$position_defs = [
			'top' => [
				'title' => '文章页面顶部',
				'desc' => '这些模块会在文章开头渲染（适合概览/要点）。',
			],
			'after' => [
				'title' => '文章中动态插入',
				'desc' => '这些模块会插入正文中（优先贴近最相关小节）。',
			],
			'bottom' => [
				'title' => '文章页面底部',
				'desc' => '这些模块会在文章末尾展示（适合 FAQ/参考资料等）。',
			],
			'code' => [
				'title' => '代码层（不直接展示）',
				'desc' => '不直接渲染在文章中：结构化输出/稳定锚点等。',
			],
		];

		$grouped = [
			'top' => [],
			'after' => [],
			'bottom' => [],
			'code' => [],
		];
		foreach ($order as $mid) {
			$pos = (string)($all_modules[$mid]['position'] ?? 'bottom');
			if (!isset($grouped[$pos])) {
				$pos = 'bottom';
			}
			$grouped[$pos][] = $mid;
		}

		echo '<div id="cr-modules-cards" class="cr-modules-cards">';

		foreach ($position_defs as $pos => $def) {
			$ids = $grouped[$pos] ?? [];
			if ($ids === []) {
				continue;
			}

			echo '<div class="cr-modules-group cr-modules-group--' . esc_attr($pos) . '">';
			echo '<h4 class="cr-modules-group__title">' . esc_html((string)$def['title']) . '</h4>';
			echo '<p class="description cr-modules-group__desc">' . esc_html((string)$def['desc']) . '</p>';
			echo '<div class="cr-modules-group__grid">';

			foreach ($ids as $module_id) {
			$cfg = is_array($modules_cfg[$module_id] ?? null) ? $modules_cfg[$module_id] : [];
			$is_content = Registry::is_content_module($module_id);
			$uses_ai = $is_content && !in_array($module_id, ['last_updated', 'toc'], true);

			$plan = Capabilities::module_plan($module_id);
			$is_pro_module = $is_content && $plan === Capabilities::PLAN_PRO;
			$locked = $is_content && !Capabilities::is_module_available($module_id, $options);
			$disabled = ((!$is_pro && $locked) || $readonly) ? ' disabled' : '';

			$enabled = !empty($cfg['enabled']);
			$label = (string)($cfg['label'] ?? $module_id);
			$prompt = (string)($cfg['prompt'] ?? '');
			$intro = (string)($all_modules[$module_id]['intro'] ?? '');
			if ($intro === '') {
				$intro = '用于增强页面的 GEO 内容结构与可读性。';
			}

			$card_class = 'cr-module-card' . ((!$is_pro && $locked) ? ' cr-module-card--locked' : '') . (!$enabled ? ' cr-module-card--disabled' : '');
			echo '<div class="' . esc_attr($card_class) . '" data-module-id="' . esc_attr($module_id) . '" data-module-label="' . esc_attr($label) . '" data-module-intro="' . esc_attr($intro) . '">';

			echo '<div class="cr-module-card__head">';
			echo '<div class="cr-module-card__title">';
			echo '<strong>' . esc_html($label !== '' ? $label : $module_id) . '</strong>';
			echo '<span class="cr-badge-wrap">';
			echo $is_pro_module ? Badges::pro('该模块为 Pro 模块') : '';
			echo $uses_ai ? Badges::ai('生成将消耗 AI Token') : '';
			echo '</span>';
			echo '</div>';

			echo '<div class="cr-module-card__actions">';
			echo '<label class="cr-module-card__toggle">';
			echo '<input type="checkbox" name="modules[' . esc_attr($module_id) . '][enabled]" value="1"' . checked($enabled, true, false) . $disabled . ' />';
			echo '<span>启用</span>';
			echo '</label>';
			echo '</div>';
			echo '</div>'; // head

			echo '<div class="cr-module-card__body">';
			echo '<div class="cr-module-card__content">';

			echo '<div class="cr-module-card__intro">' . esc_html($intro) . '</div>';

			// 摘要信息（卡片内高密度展示）
			echo '<div class="cr-module-card__summary">';

			echo '<div class="cr-module-card__summary-item">';
			echo '<div class="cr-module-card__summary-k">AI</div>';
			if ($is_content && $uses_ai) {
				echo '<div class="cr-module-card__summary-v">启用后消耗 Token</div>';
			} elseif ($is_content && !$uses_ai) {
				echo '<div class="cr-module-card__summary-v">不走 AI（本地）</div>';
			} else {
				echo '<div class="cr-module-card__summary-v">—</div>';
			}
			echo '</div>';

			echo '<div class="cr-module-card__summary-item">';
			echo '<div class="cr-module-card__summary-k">规则</div>';
			if ($module_id === 'faq') {
				$min = (int)($cfg['min_rules']['faq_min_items'] ?? 3);
				echo '<div class="cr-module-card__summary-v">最少条数：' . esc_html((string)$min) . '</div>';
			} elseif ($module_id === 'toc') {
				$toc = is_array($cfg['toc'] ?? null) ? $cfg['toc'] : [];
				$open = !empty($toc['default_open']);
				$style = sanitize_key((string)($toc['list_style'] ?? 'unordered'));
				$style_label = $style === 'ordered' ? '有序' : '无序';
				$max = (int)($toc['max_level'] ?? 4);
				$max = max(2, min(5, $max));
				echo '<div class="cr-module-card__summary-v">默认' . ($open ? '展开' : '折叠') . '｜' . esc_html($style_label) . '｜到 H' . esc_html((string)$max) . '</div>';
			} elseif ($module_id === 'summary') {
				$mode = sanitize_key((string)($cfg['summary_excerpt'] ?? 'off'));
				if (!in_array($mode, ['off', 'force', 'compat'], true)) {
					$mode = 'off';
				}
				$label_map = [
					'off' => '不写入摘要',
					'compat' => '摘要为空时写入',
					'force' => '强制写入摘要',
				];
				echo '<div class="cr-module-card__summary-v">' . esc_html($label_map[$mode] ?? '—') . '</div>';
			} elseif ($module_id === 'table') {
				echo '<div class="cr-module-card__summary-v">优先匹配小节，兜底第 3 段</div>';
			} else {
				echo '<div class="cr-module-card__summary-v">—</div>';
			}
			echo '</div>';

			echo '<div class="cr-module-card__summary-item">';
			echo '<div class="cr-module-card__summary-k">提示词</div>';
			if ($is_content && $uses_ai) {
				echo '<div class="cr-module-card__summary-v">' . ($prompt !== '' ? '已自定义' : '默认') . '</div>';
			} else {
				echo '<div class="cr-module-card__summary-v">—</div>';
			}
			echo '</div>';

			echo '</div>'; // summary

			echo '<details class="cr-module-card__details">';
			echo '<summary>配置</summary>';

			echo '<div class="cr-module-card__details-body">';

			echo '<div class="cr-module-field">';
			echo '<label>显示名称</label>';
			echo '<input type="text" name="modules[' . esc_attr($module_id) . '][label]" value="' . esc_attr($label) . '"' . $disabled . ' />';
			echo '<p class="description">用于前端模块标题与编辑器面板显示。</p>';
			echo '</div>';

			if ($module_id === 'faq') {
				$min = (int)($cfg['min_rules']['faq_min_items'] ?? 3);
				echo '<div class="cr-module-field">';
				echo '<label>规则</label>';
				echo '<label>最少条数 <input type="number" min="1" max="20" name="modules[' . esc_attr($module_id) . '][min_rules][faq_min_items]" value="' . esc_attr((string)$min) . '"' . $disabled . ' /></label>';
				echo '<p class="description">条数不足会在检查中提示。</p>';
				echo '</div>';
			} elseif ($module_id === 'toc') {
				$toc = is_array($cfg['toc'] ?? null) ? $cfg['toc'] : [];
				$open = !empty($toc['default_open']);
				$style = sanitize_key((string)($toc['list_style'] ?? 'unordered'));
				if (!in_array($style, ['unordered', 'ordered'], true)) {
					$style = 'unordered';
				}
				$max = (int)($toc['max_level'] ?? 4);
				$max = max(2, min(5, $max));

				echo '<div class="cr-module-field">';
				echo '<label>规则</label>';

				echo '<div style="margin-bottom:6px;">';
				echo '<input type="hidden" name="modules[' . esc_attr($module_id) . '][toc][default_open]" value="0"' . $disabled . ' />';
				echo '<label><input type="checkbox" name="modules[' . esc_attr($module_id) . '][toc][default_open]" value="1"' . checked($open, true, false) . $disabled . ' /> 默认展开</label>';
				echo '</div>';

				echo '<div style="margin-bottom:6px;">';
				echo '<label>目录样式</label> ';
				echo '<select name="modules[' . esc_attr($module_id) . '][toc][list_style]"' . $disabled . '>';
				echo '<option value="unordered"' . selected($style, 'unordered', false) . '>无序（圆点）</option>';
				echo '<option value="ordered"' . selected($style, 'ordered', false) . '>有序（数字）</option>';
				echo '</select>';
				echo '</div>';

				echo '<div style="margin-bottom:6px;">';
				echo '<label>最大标题层级</label> ';
				echo '<select name="modules[' . esc_attr($module_id) . '][toc][max_level]"' . $disabled . '>';
				foreach ([4 => '到 H4（推荐）', 5 => '到 H5（更细）'] as $k => $t) {
					echo '<option value="' . esc_attr((string)$k) . '"' . selected($max, $k, false) . '>' . esc_html($t) . '</option>';
				}
				echo '</select>';
				echo '</div>';

				echo '<p class="description">提示：目录只会读取正文里的标题（H2-H5），不包含文章标题（H1）。</p>';
				echo '</div>';
			} elseif ($module_id === 'summary') {
				$mode = sanitize_key((string)($cfg['summary_excerpt'] ?? 'off'));
				if (!in_array($mode, ['off', 'force', 'compat'], true)) {
					$mode = 'off';
				}
				echo '<div class="cr-module-field">';
				echo '<label>规则</label>';
				echo '<label>写入文章摘要</label>';
				echo '<select name="modules[' . esc_attr($module_id) . '][summary_excerpt]"' . $disabled . '>';
				foreach (['off' => '不写入', 'compat' => '仅在文章摘要为空时写入（摘要不为空则不生成）', 'force' => '强制写入（覆盖已有摘要）'] as $k => $t) {
					echo '<option value="' . esc_attr($k) . '"' . selected($mode, $k, false) . '>' . esc_html($t) . '</option>';
				}
				echo '</select>';
				echo '<p class="description">用于让主题/SEO 插件更容易复用摘要内容。</p>';
				echo '</div>';
			} elseif ($module_id === 'table') {
				echo '<div class="cr-module-field">';
				echo '<label>插入策略</label>';
				echo '<div class="cr-module-card__muted">优先：匹配最相关的小节标题，插在该标题后的第一段落后；兜底：插在第 3 段落后（避免出现在文章开头）；最后兜底：追加到正文末尾。</div>';
				echo '<p class="description">提示：该策略用于让表格更自然地贴近对应的小节；如你的主题没有标准段落/标题结构，可能会退化为兜底策略。</p>';
				echo '</div>';
			}

			echo '<div class="cr-module-field">';
			echo '<label>提示词（可选） <span class="cr-badge-wrap">' . ($uses_ai ? Badges::ai('生成将消耗 AI Token') : '') . '</span></label>';
			if ($is_content && $uses_ai) {
				echo '<textarea rows="4" name="modules[' . esc_attr($module_id) . '][prompt]" placeholder="可选：为该模块追加额外要求（不会改变模块输出结构）"' . $disabled . '>' . esc_textarea($prompt) . '</textarea>';
				echo '<p class="description">提示：这是“补充要求”，会在生成时追加到你的提示后面。</p>';
			} elseif ($is_content && !$uses_ai) {
				echo '<input type="hidden" name="modules[' . esc_attr($module_id) . '][prompt]" value=""' . $disabled . ' />';
				echo '<div class="cr-module-card__muted">不走 AI</div>';
			} else {
				echo '<input type="hidden" name="modules[' . esc_attr($module_id) . '][prompt]" value=""' . $disabled . ' />';
				echo '<div class="cr-module-card__muted">—</div>';
			}
			echo '</div>';

			if (!$is_pro && $locked) {
				echo '<p class="description">Free：该模块为 Pro，已锁定不可启用。</p>';
			}

			echo '</div>'; // details-body
			echo '</details>';

			echo '</div>'; // content

			if (!$is_pro && $locked) {
				$upgrade_url = trim((string)apply_filters('cr_pro_buy_url', 'https://xmtbx.com/content-ready'));
				if ($upgrade_url === '') {
					$upgrade_url = admin_url('admin.php?page=content-ready-settings&tab=license');
				}
				$upgrade_target = ' target="_blank" rel="noopener noreferrer"';
				echo '<div class="cr-module-card__overlay">';
				echo '<div class="cr-module-card__overlay-title">解锁 Pro 模块</div>';
				echo '<div class="cr-module-card__overlay-desc">' . esc_html($intro) . '</div>';
				echo '<a class="button button-primary" href="' . esc_url($upgrade_url) . '"' . $upgrade_target . '>升级到 Pro 版本</a>';
				echo '</div>';
			}

			echo '</div>'; // body
			echo '</div>'; // card
		}

			echo '</div>'; // grid
			echo '</div>'; // group
		}

		echo '</div>'; // cards
	}
}

