<?php

declare(strict_types=1);

namespace ContentReady\Admin\SettingsSections;

use ContentReady\Admin\Ui\Badges;
use ContentReady\Licensing\Capabilities;

final class AutomationSection
{
	public static function render(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);

		$gen = is_array($options['generation'] ?? null) ? $options['generation'] : [];
		$auto_on_publish = !empty($gen['auto_geo_on_publish']);

		$backfill_enabled = !empty($gen['backfill_enabled']);
		$backfill_mode = sanitize_key((string)($gen['backfill_mode'] ?? 'once')); // once | daily
		if (!in_array($backfill_mode, ['once', 'daily'], true)) {
			$backfill_mode = 'once';
		}

		$daily_limit = (int)($gen['backfill_daily_limit'] ?? 10);
		$daily_limit = max(1, min(200, $daily_limit));

		$hour = (int)($gen['backfill_hour'] ?? 3);
		$hour = max(0, min(23, $hour));

		echo '<hr />';
		echo '<h2>' . esc_html__('自动化', 'content-ready') . ' <span class="cr-badge-wrap">' . Badges::pro() . Badges::ai() . '</span></h2>';
		echo '<p class="description">用于“发布后自动 GEO”和“老文章批量 GEO”。为避免卡发布/影响访问，默认采用异步调度（WP-Cron）。</p>';
		echo '<p class="description"><strong>注意：</strong>自动化会消耗 AI Token（产生费用），建议先用少量文章验证效果与成本后再开启；站点访问量较低时，WP-Cron 触发可能不稳定，建议配合服务器 Cron（可选）。</p>';

		if (!$is_pro) {
			echo '<div class="notice notice-info inline"><p>当前为 Free：自动化功能需升级 Pro 解锁。</p></div>';
		}

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row">文章发布时自动 GEO <span class="cr-badge-wrap">' . Badges::pro() . Badges::ai() . '</span></th><td>';
		if ($is_pro) {
			// Pro/AI 标识：放在标题后，不放在描述区（见文档约定）
			echo '<label><input type="checkbox" name="generation_auto_geo_on_publish" value="1"' . checked($auto_on_publish, true, false) . ' /> 启用</label>';
			echo '<p class="description">仅在你主动开启时生效；执行为异步调度，不阻塞发布/更新。</p>';
		} else {
			echo '<input type="hidden" name="generation_auto_geo_on_publish" value="0" />';
			echo '<span style="color:#6b7280;">Pro 功能</span>';
		}
		echo '</td></tr>';

		echo '<tr><th scope="row">老文章自动回填（批量 GEO） <span class="cr-badge-wrap">' . Badges::pro() . Badges::ai() . '</span></th><td>';
		if ($is_pro) {
			// Pro/AI 标识：放在标题后，不放在描述区（见文档约定）
			echo '<label><input type="checkbox" name="generation_backfill_enabled" value="1"' . checked($backfill_enabled, true, false) . ' /> 启用</label>';
			echo '<p class="description">支持一次性跑完（适合文章量少）或每日定量回填（适合性能一般/访问量大站点，建议设置在凌晨）。回填默认“仅补缺失模块 + 跳过生成后被手动编辑过的模块”，避免覆盖你的人工修改。</p>';

			echo '<p style="margin:8px 0 0;">运行方式：';
			echo '<select name="generation_backfill_mode">';
			foreach (['once' => '一次性跑完（后台异步分批直到完成）', 'daily' => '每日定量回填'] as $k => $label) {
				echo '<option value="' . esc_attr($k) . '"' . selected($backfill_mode, $k, false) . '>' . esc_html($label) . '</option>';
			}
			echo '</select></p>';

			echo '<p style="margin:8px 0 0;">每日回填篇数：';
			echo '<input type="number" min="1" max="200" style="width:96px" name="generation_backfill_daily_limit" value="' . esc_attr((string)$daily_limit) . '" />';
			echo '</p>';

			echo '<p style="margin:8px 0 0;">建议执行时段：';
			echo '<select name="generation_backfill_hour">';
			for ($h = 0; $h <= 23; $h++) {
				$label = str_pad((string)$h, 2, '0', STR_PAD_LEFT) . ':00';
				echo '<option value="' . esc_attr((string)$h) . '"' . selected($hour, $h, false) . '>' . esc_html($label) . '</option>';
			}
			echo '</select>';
			echo '<span style="color:#6b7280;margin-left:6px;">（本地站点时间）</span>';
			echo '</p>';
		} else {
			echo '<input type="hidden" name="generation_backfill_enabled" value="0" />';
			echo '<input type="hidden" name="generation_backfill_mode" value="once" />';
			echo '<input type="hidden" name="generation_backfill_daily_limit" value="10" />';
			echo '<input type="hidden" name="generation_backfill_hour" value="3" />';
			echo '<span style="color:#6b7280;">Pro 功能</span>';
		}
		echo '</td></tr>';

		echo '</tbody></table>';
	}
}
