<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Licensing\Capabilities;
use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\PageTypes\Config as PageTypesConfig;
use ContentReady\Util\Locale;

final class SettingsSaver
{
	public static function handle_save(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die('Forbidden');
		}
		check_admin_referer('cr_save_settings');

		$options = wp_parse_args(Settings::get_raw_options(), Settings::default_options());
		$options['modules_order'] = ModulesConfig::default_order();
		$options['modules'] = ModulesConfig::normalize_modules_config($options['modules'] ?? null);
		$options['page_types'] = PageTypesConfig::normalize_page_types($options['page_types'] ?? null);
		$options['config_version'] = Settings::CONFIG_VERSION;

		$tab = isset($_POST['cr_settings_tab']) ? sanitize_key((string)$_POST['cr_settings_tab']) : 'basic';

		switch ($tab) {
			case 'ai_privacy':
				$options['ai']['base_url'] = isset($_POST['ai_base_url']) ? esc_url_raw((string)$_POST['ai_base_url']) : '';
				$options['ai']['api_key'] = isset($_POST['ai_api_key']) ? sanitize_text_field((string)$_POST['ai_api_key']) : '';
				$options['ai']['model'] = isset($_POST['ai_model']) ? sanitize_text_field((string)$_POST['ai_model']) : '';
				$ai_mode = isset($_POST['ai_mode']) ? sanitize_key((string)$_POST['ai_mode']) : 'standard';
				if (!in_array($ai_mode, ['quick', 'standard', 'high'], true)) {
					$ai_mode = 'standard';
				}
				$options['ai']['mode'] = $ai_mode;

				$send_scope = isset($_POST['privacy_send_scope']) ? sanitize_key((string)$_POST['privacy_send_scope']) : 'full';
				if (!in_array($send_scope, ['full', 'title_summary', 'selected_modules'], true)) {
					$send_scope = 'full';
				}
				$options['privacy']['send_scope'] = $send_scope;
				break;

			case 'modules_output':
				$options['frontend']['inject_enabled'] = !empty($_POST['frontend_inject_enabled']);
				$options['frontend']['css_enabled'] = !empty($_POST['frontend_css_enabled']);

				$options['structured_output']['enabled'] = !empty($_POST['structured_output_enabled']);

				// 新版：模块调度策略（off/adaptive/lightweight）
				if (isset($_POST['generation_dispatch_mode'])) {
					$dispatch = sanitize_key((string)$_POST['generation_dispatch_mode']);
					if (!in_array($dispatch, ['off', 'adaptive', 'lightweight'], true)) {
						$dispatch = 'off';
					}
					$options['generation']['dispatch_mode'] = $dispatch;
					$options['generation']['adaptive_enabled'] = ($dispatch === 'adaptive');
				} else {
					// 兼容旧版：自适应开关（布尔）
					$options['generation']['adaptive_enabled'] = !empty($_POST['generation_adaptive_enabled']);
					$options['generation']['dispatch_mode'] = $options['generation']['adaptive_enabled'] ? 'adaptive' : 'off';
				}

				if (isset($_POST['structured_output_mode'])) {
					$mode = sanitize_key((string)$_POST['structured_output_mode']);
					if (!in_array($mode, ['auto', 'force', 'off'], true)) {
						$mode = 'auto';
					}
					$options['structured_output']['mode'] = $mode;
				}

				$posted_modules = isset($_POST['modules']) && is_array($_POST['modules']) ? $_POST['modules'] : null;
				if ($posted_modules !== null) {
					$options['modules'] = ModulesConfig::sanitize_modules_config_partial($posted_modules, $options['modules'] ?? null);
				}
				break;

			case 'page_types':
				$is_pro = Capabilities::is_pro($options);
				$old_page_types = $options['page_types'] ?? null;
				$options['page_types'] = PageTypesConfig::sanitize_from_post($_POST, $options);

				// Free 保存时：保留“未展示”的 Pro 模块配置（避免被表单覆盖清空）。
				if (!$is_pro) {
					$existing = PageTypesConfig::normalize_page_types($old_page_types);
					$all_content = array_values(array_filter(ModulesConfig::default_order(), [\ContentReady\Modules\Registry::class, 'is_content_module']));

					$preserved = [];
					foreach ($existing as $row) {
						if (is_array($row) && isset($row['id'])) {
							$preserved[(string)$row['id']] = is_array($row['modules'] ?? null) ? $row['modules'] : [];
						}
					}

					$out = [];
					foreach ($options['page_types'] as $row) {
						if (!is_array($row)) {
							continue;
						}
						$id = (string)($row['id'] ?? '');
						$mods = is_array($row['modules'] ?? null) ? $row['modules'] : [];
						$old = is_array($preserved[$id] ?? null) ? $preserved[$id] : [];
						$keep = [];
						foreach ($old as $mid) {
							$mid = sanitize_key((string)$mid);
							if (Capabilities::module_plan($mid) === Capabilities::PLAN_PRO) {
								$keep[$mid] = true;
							}
						}

						$final = [];
						$want = [];
						foreach ($mods as $mid) {
							$want[sanitize_key((string)$mid)] = true;
						}
						foreach ($all_content as $mid) {
							if (isset($want[$mid]) || isset($keep[$mid])) {
								$final[] = $mid;
							}
						}

						$row['modules'] = $final;
						$out[] = $row;
					}

					$options['page_types'] = $out;
				}
				break;

			case 'automation':
				$options['generation']['auto_geo_on_publish'] = !empty($_POST['generation_auto_geo_on_publish']);
				$options['generation']['backfill_enabled'] = !empty($_POST['generation_backfill_enabled']);
				$bf_mode = isset($_POST['generation_backfill_mode']) ? sanitize_key((string)$_POST['generation_backfill_mode']) : 'once';
				if (!in_array($bf_mode, ['once', 'daily'], true)) {
					$bf_mode = 'once';
				}
				$options['generation']['backfill_mode'] = $bf_mode;
				$bf_limit = isset($_POST['generation_backfill_daily_limit']) ? (int)$_POST['generation_backfill_daily_limit'] : 10;
				$options['generation']['backfill_daily_limit'] = max(1, min(200, $bf_limit));
				$bf_hour = isset($_POST['generation_backfill_hour']) ? (int)$_POST['generation_backfill_hour'] : 3;
				$options['generation']['backfill_hour'] = max(0, min(23, $bf_hour));
				break;

			case 'license':
				$license_key = isset($_POST['license_key']) ? sanitize_text_field((string)$_POST['license_key']) : '';
				$options['license']['key'] = $license_key;
				$options['license']['status'] = $license_key === '' ? 'unlicensed' : 'unverified';
				$options['license']['plan'] = 'free';
				$options['license']['checked_at'] = 0;
				$options['license']['expires_at'] = 0;
				$options['license']['grace_until'] = 0;
				$options['license']['last_error'] = '';
				$options['license']['last_message'] = '';
				break;

			case 'advanced':
				if (isset($_POST['structured_output_mode'])) {
					$mode = sanitize_key((string)$_POST['structured_output_mode']);
					if (!in_array($mode, ['auto', 'force', 'off'], true)) {
						$mode = 'auto';
					}
					$options['structured_output']['mode'] = $mode;
				}

				$options['uninstall']['cleanup'] = !empty($_POST['uninstall_cleanup']);
				break;

			case 'basic':
			default:
				$options['site']['display_name'] = isset($_POST['site_display_name']) ? sanitize_text_field((string)$_POST['site_display_name']) : '';
				$options['site']['logo_url'] = isset($_POST['site_logo_url']) ? esc_url_raw((string)$_POST['site_logo_url']) : '';
				$locale = isset($_POST['site_locale']) ? Locale::normalize_bcp47((string)$_POST['site_locale']) : '';
				$options['site']['locale'] = $locale;

				$df_mode = isset($_POST['site_date_format_mode']) ? sanitize_key((string)$_POST['site_date_format_mode']) : 'wp';
				if ($df_mode === 'custom') {
					$options['site']['date_format'] = isset($_POST['site_date_format_custom']) ? sanitize_text_field((string)$_POST['site_date_format_custom']) : 'wp';
					if ((string)$options['site']['date_format'] === '') {
						$options['site']['date_format'] = 'wp';
					}
				} else {
					$options['site']['date_format'] = 'wp';
				}

				$author_mode = isset($_POST['site_author_mode']) ? sanitize_key((string)$_POST['site_author_mode']) : 'wp';
				if (!in_array($author_mode, ['wp', 'brand'], true)) {
					$author_mode = 'wp';
				}
				$options['site']['author_mode'] = $author_mode;
				$options['site']['brand_author_name'] = isset($_POST['site_brand_author_name']) ? sanitize_text_field((string)$_POST['site_brand_author_name']) : '';

				$options['checks']['auto_on_publish'] = !empty($_POST['checks_auto_on_publish']);

				if (!is_array($options['ui'] ?? null)) {
					$options['ui'] = [];
				}
				$options['ui']['editor_entry_label'] = isset($_POST['ui_editor_entry_label'])
					? trim(sanitize_text_field((string)$_POST['ui_editor_entry_label']))
					: trim((string)($options['ui']['editor_entry_label'] ?? ''));
				break;
		}

		Settings::update_options($options);

		$base = admin_url('admin.php');
		if ($tab === 'automation') {
			$url = add_query_arg(['page' => 'content-ready-automation', 'cr_notice' => 'cr_saved'], $base);
		} else {
			$url = add_query_arg(['page' => 'content-ready-settings', 'tab' => $tab, 'cr_notice' => 'cr_saved'], $base);
		}
		wp_safe_redirect($url);
		exit;
	}
}
