<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Admin\SettingsSections\ImportExportSection;
use ContentReady\Admin\SettingsSections\ModulesSection;
use ContentReady\Admin\SettingsSections\PageTypesSection;
use ContentReady\Admin\Ui\Badges;
use ContentReady\Admin\Ui\PlanCompare;
use ContentReady\Licensing\Capabilities;
use ContentReady\Updates\Updater;
use ContentReady\Util\Locale;

final class SettingsPage
{
	/**
	 * @param string $active_tab basic|ai_privacy|modules_output|page_types|license|import_export|advanced
	 */
	public static function render(string $notice, array $options, string $active_tab): void
	{
		echo '<div class="wrap">';
		echo '<h1>' . esc_html__('Content Ready｜内容就绪', 'content-ready') . '</h1>';

		self::render_notice($notice);
		self::render_tabs($active_tab);
		self::render_tab_content($active_tab, $options);

		echo '</div>';
	}

	private static function tabs(): array
	{
		return [
			'basic' => '基础设置',
			'ai_privacy' => 'AI 与隐私',
			'modules_output' => '模块与输出',
			'page_types' => '页面类型规则',
			'license' => '许可证（Pro）',
			'import_export' => '导入导出',
			'advanced' => '高级/维护',
		];
	}

	private static function render_tabs(string $active_tab): void
	{
		$tabs = self::tabs();
		if (!isset($tabs[$active_tab])) {
			$active_tab = 'basic';
		}

		echo '<h2 class="nav-tab-wrapper" style="margin-top:16px;">';
		foreach ($tabs as $k => $label) {
			$url = add_query_arg(
				['page' => 'content-ready-settings', 'tab' => $k],
				admin_url('admin.php')
			);
			$cls = 'nav-tab' . ($k === $active_tab ? ' nav-tab-active' : '');
			echo '<a class="' . esc_attr($cls) . '" href="' . esc_url($url) . '">' . esc_html($label) . '</a>';
		}
		echo '</h2>';
	}

	private static function render_tab_content(string $active_tab, array $options): void
	{
		$tabs = self::tabs();
		if (!isset($tabs[$active_tab])) {
			$active_tab = 'basic';
		}

		switch ($active_tab) {
			case 'ai_privacy':
				self::render_tab_ai_privacy($options);
				return;
			case 'modules_output':
				self::render_tab_modules_output($options);
				return;
			case 'page_types':
				self::render_tab_page_types($options);
				return;
			case 'license':
				self::render_tab_license($options);
				return;
			case 'import_export':
				self::render_tab_import_export();
				return;
			case 'advanced':
				self::render_tab_advanced($options);
				return;
			case 'basic':
			default:
				self::render_tab_basic($options);
				return;
		}
	}

	private static function render_notice(string $notice): void
	{
		if ($notice === '') {
			return;
		}

		if ($notice === 'cr_update_checked') {
			$status = isset($_GET['cr_update_status']) ? sanitize_key((string)$_GET['cr_update_status']) : '';
			$remote = isset($_GET['cr_update_remote']) ? sanitize_text_field((string)$_GET['cr_update_remote']) : '';
			$checked_at = isset($_GET['cr_update_checked_at']) ? (int)$_GET['cr_update_checked_at'] : 0;
			$current = defined('CR_VERSION') ? (string)CR_VERSION : '';

			$cls = 'notice notice-info';
			$msg = '已检查更新。';
			if ($status === 'available' && $remote !== '' && $current !== '') {
				$cls = 'notice notice-warning';
				$msg = '已检查更新：发现新版本 ' . $remote . '（当前 ' . $current . '）。你可在「插件」页一键更新，或打开更新日志查看说明与下载链接。';
			} elseif ($status === 'latest' && $current !== '') {
				$msg = '已检查更新：当前已是最新版本（' . $current . '）。';
			} elseif ($status === 'failed') {
				$cls = 'notice notice-error';
				$msg = '检查更新失败：未能获取最新版本信息，请稍后重试。';
			}

			if ($checked_at > 0) {
				$msg .= '（检查时间：' . wp_date('Y-m-d H:i', $checked_at) . '）';
			}

			echo '<div class="' . esc_attr($cls) . '"><p>' . esc_html($msg) . '</p></div>';
			return;
		}

		$messages = [
			'cr_saved' => '设置已保存。',
			'cr_license_verified' => '许可证校验成功：已解锁 Pro。',
			'cr_license_verify_failed' => '许可证校验未通过：请检查 key 与域名绑定，或稍后重试。',
			'cr_import_success' => '导入成功，配置已生效。',
			'cr_import_missing_file' => '导入失败：未选择文件。',
			'cr_import_invalid_json' => '导入失败：JSON 解析失败。',
			'cr_import_version_mismatch' => '导入失败：配置版本不匹配（可能来自旧版本导出文件）。',
		];
		$msg = $messages[$notice] ?? $notice;
		echo '<div class="notice notice-info"><p>' . esc_html($msg) . '</p></div>';
	}

	private static function open_form(string $tab): void
	{
		echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
		wp_nonce_field('cr_save_settings');
		echo '<input type="hidden" name="action" value="cr_save_settings" />';
		echo '<input type="hidden" name="cr_settings_tab" value="' . esc_attr($tab) . '" />';
	}

	private static function close_form(string $label = '保存设置'): void
	{
		submit_button($label);
		echo '</form>';
	}

	private static function render_badge_legend(): void
	{
		echo '<p class="description" style="margin:12px 0 0;">';
		echo '<span class="cr-badge-wrap">标记：' . Badges::pro('Pro 功能（即使已解锁也会常驻标注）') . Badges::ai('将消耗 AI Token') . '</span>';
		echo '</p>';
	}

	private static function render_tab_basic(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);
		$site = is_array($options['site'] ?? null) ? $options['site'] : [];
		$ui = is_array($options['ui'] ?? null) ? $options['ui'] : [];

		echo '<h2 style="margin-top:16px;">' . esc_html__('基础设置', 'content-ready') . '</h2>';
		self::open_form('basic');

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row"><label for="cr_site_display_name">网站名称（展示用）</label></th><td>';
		echo '<input class="regular-text" id="cr_site_display_name" name="site_display_name" value="' . esc_attr((string)($site['display_name'] ?? '')) . '" />';
		echo '<p class="description">用于结构化输出的 publisher/name；留空则默认读取 WordPress「设置 → 常规」中的站点标题（不修改全站设置）。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_site_logo_url">网站 Logo URL（可选）</label></th><td>';
		$logo_url = (string)($site['logo_url'] ?? '');
		echo '<div style="display:flex;gap:8px;align-items:center;max-width:760px;">';
		echo '<input class="regular-text" id="cr_site_logo_url" name="site_logo_url" value="' . esc_attr($logo_url) . '" placeholder="https://example.com/logo.png" style="flex:1;min-width:220px;" />';
		echo '<button type="button" class="button" id="cr_site_logo_pick">从媒体库选择</button>';
		echo '<button type="button" class="button" id="cr_site_logo_clear">清除</button>';
		echo '</div>';
		echo '<div id="cr_site_logo_preview" style="margin-top:8px;">';
		if ($logo_url !== '') {
			echo '<img src="' . esc_url($logo_url) . '" alt="" style="max-width:120px;height:auto;border:1px solid #e5e7eb;border-radius:8px;padding:6px;background:#fff;" />';
		}
		echo '</div>';
		echo '<p class="description">用于结构化输出 publisher/logo（ImageObject.url）。推荐尺寸：正方形 ≥ 512×512（清晰度更好）。如未填写：优先使用“站点身份 → 站点图标”；若仍无则尝试站点根目录的 favicon.ico；都没有则不输出 logo。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_site_locale">默认语言 / 地区</label></th><td>';
		$selected_locale = (string)($site['locale'] ?? '');
		$current = Locale::wp_locale_to_bcp47((string)get_locale());
		$available = function_exists('get_available_languages') ? get_available_languages() : [];
		$raw_list = array_values(array_unique(array_merge([(string)get_locale()], is_array($available) ? $available : [])));
		$map = [];
		foreach ($raw_list as $wp_locale) {
			$tag = Locale::wp_locale_to_bcp47((string)$wp_locale);
			if ($tag === '') {
				continue;
			}
			$map[$tag] = (string)$wp_locale;
		}
		ksort($map);

		echo '<select id="cr_site_locale" name="site_locale">';
		echo '<option value="">' . esc_html__('跟随站点语言（WordPress）', 'content-ready') . '</option>';
		foreach ($map as $tag => $wp_locale) {
			$label = $tag . ($wp_locale !== '' ? ('（' . $wp_locale . '）') : '');
			echo '<option value="' . esc_attr($tag) . '"' . selected($selected_locale, $tag, false) . '>' . esc_html($label) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">用于结构化输出 inLanguage。当前站点语言：' . esc_html($current !== '' ? $current : (string)get_locale()) . '</p>';
		echo '</td></tr>';

		$df = (string)($site['date_format'] ?? 'wp');
		$df_mode = $df === 'wp' ? 'wp' : 'custom';
		echo '<tr><th scope="row">日期展示格式</th><td>';
		echo '<label><input type="radio" name="site_date_format_mode" value="wp"' . checked($df_mode, 'wp', false) . ' /> 使用 WordPress 日期格式（' . esc_html((string)get_option('date_format')) . '）</label><br />';
		echo '<label><input type="radio" name="site_date_format_mode" value="custom"' . checked($df_mode, 'custom', false) . ' /> 自定义</label><br />';
		echo '<input class="regular-text" name="site_date_format_custom" value="' . esc_attr($df_mode === 'custom' ? $df : '') . '" placeholder="Y-m-d" />';
		echo '<p class="description">用于“可见更新时间”等模块显示；格式为 PHP date() 语法。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">默认作者显示策略 ' . Badges::pro('作者策略为 Pro 功能') . '</th><td>';
		if ($is_pro) {
			$author_mode = (string)($site['author_mode'] ?? 'wp');
			if (!in_array($author_mode, ['wp', 'brand'], true)) {
				$author_mode = 'wp';
			}
			echo '<label><input type="radio" name="site_author_mode" value="wp"' . checked($author_mode, 'wp', false) . ' /> WordPress 作者</label><br />';
			echo '<label><input type="radio" name="site_author_mode" value="brand"' . checked($author_mode, 'brand', false) . ' /> 品牌 / 团队</label><br />';
			echo '<input class="regular-text" name="site_brand_author_name" value="' . esc_attr((string)($site['brand_author_name'] ?? '')) . '" placeholder="品牌作者名称（可选）" />';
			echo '<p class="description">用于结构化输出 author。你也可以在单篇文章中覆盖作者策略（编辑器侧）。</p>';
		} else {
			echo '<span style="color:#6b7280;">Free：固定使用 WordPress 作者。</span>';
		}
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_ui_editor_entry_label">编辑器入口名称</label></th><td>';
		$editor_entry_label = trim(sanitize_text_field((string)($ui['editor_entry_label'] ?? '')));
		echo '<input class="regular-text" id="cr_ui_editor_entry_label" name="ui_editor_entry_label" value="' . esc_attr($editor_entry_label) . '" placeholder="内容就绪" />';
		echo '<p class="description">用于 Gutenberg 侧边栏与经典编辑器 Meta Box 的标题；建议短一点，避免编辑器侧边栏拥挤。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">发布前检查</th><td>';
		$auto = !empty($options['checks']['auto_on_publish']);
		echo '<label><input type="checkbox" name="checks_auto_on_publish" value="1"' . checked($auto, true, false) . ' /> 发布/更新时自动运行“发布前检查”（不阻塞发布）</label>';
		echo '<p class="description">发布/更新后自动生成一份检查报告（不阻塞发布），用于更早发现“模块缺失/FAQ 条数不足/重复结构化输出”等问题。仅做检查，不会自动生成模块；如需发布后自动补齐模块，请到「自动化」开启（Pro）。你也可在编辑器内随时手动运行“发布前检查”查看详情。</p>';
		echo '</td></tr>';

		echo '</tbody></table>';
		self::close_form();
	}

	private static function render_tab_ai_privacy(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);
		$ai = is_array($options['ai'] ?? null) ? $options['ai'] : [];

		echo '<div class="notice notice-info inline" style="margin-top:16px;"><p>';
		echo esc_html__('Content Ready 不提供 AI Key：AI 生成基于你自有的接口配置。', 'content-ready');
		echo '</p></div>';

		echo '<h2 style="margin-top:16px;">' . esc_html__('AI 与隐私', 'content-ready') . '</h2>';
		self::open_form('ai_privacy');

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row"><label for="cr_ai_base_url">AI 接口地址（API Base）</label></th><td>';
		echo '<input class="regular-text" id="cr_ai_base_url" name="ai_base_url" value="' . esc_attr((string)($ai['base_url'] ?? '')) . '" placeholder="https://api.example.com/v1" />';
		echo '<p class="description">支持 OpenAI 兼容接口；请填写你购买/自建的网关地址。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_ai_api_key">AI Key</label></th><td>';
		echo '<input class="regular-text" id="cr_ai_api_key" name="ai_api_key" value="' . esc_attr((string)($ai['api_key'] ?? '')) . '" autocomplete="off" />';
		echo '<p class="description">用于调用你的 AI 接口。注意：这会影响你的 Token 消耗与费用。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row"><label for="cr_ai_model">模型</label></th><td>';
		echo '<input class="regular-text" id="cr_ai_model" name="ai_model" value="' . esc_attr((string)($ai['model'] ?? '')) . '" placeholder="例如：gpt-4o-mini / deepseek-chat / llama-3.1-70b" />';
		echo '<p class="description">模型名称不做限定：请填写你的 AI 服务商实际支持的模型标识（不同服务商/渠道命名可能不同）。</p>';
		echo '</td></tr>';

		$ai_mode = sanitize_key((string)($ai['mode'] ?? 'standard'));
		if (!in_array($ai_mode, ['quick', 'standard', 'high'], true)) {
			$ai_mode = 'standard';
		}
		echo '<tr><th scope="row"><label for="cr_ai_mode">生成模式</label></th><td>';
		echo '<select id="cr_ai_mode" name="ai_mode">';
		foreach (['quick' => '快速（更简洁）', 'standard' => '标准（默认）', 'high' => '高质量（更全面）'] as $k => $label) {
			echo '<option value="' . esc_attr($k) . '"' . selected($ai_mode, $k, false) . '>' . esc_html($label) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">影响提示词对“输出长度/条数”的倾向：快速更省 Token、更快；高质量更全面、通常更耗时。参考：摘要约 80–260 字；TL;DR/适用范围通常 3–7 条；FAQ 通常 3–8 条（实际取决于文章长度、启用模块与模型能力）。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">隐私：发送范围 ' . Badges::pro('解锁“标题 + 短摘要”') . '</th><td>';
		$scope = (string)($options['privacy']['send_scope'] ?? 'full');
		echo '<select name="privacy_send_scope">';
		$choices = $is_pro
			? ['full' => '全文（推荐）', 'title_summary' => '标题 + 摘要', 'selected_modules' => '标题 + 短摘要（Pro）']
			: ['full' => '全文（推荐）', 'title_summary' => '标题 + 摘要'];
		foreach ($choices as $k => $label) {
			echo '<option value="' . esc_attr($k) . '"' . selected($scope, $k, false) . '>' . esc_html($label) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">用于控制发送给 AI 的上下文范围。默认推荐“全文”：更贴合文章语境、效果更稳。范围越小越省 Token/更少暴露内容，但生成效果可能更不稳定；如果你的目标主要是省 Token，优先考虑减少启用模块数量或使用「固定轻量化模块调度」。</p>';
		if (!$is_pro) {
			echo '<p class="description">Free：暂不支持“标题 + 短摘要”。</p>';
		}
		echo '</td></tr>';

		echo '</tbody></table>';
		self::close_form();
	}

	private static function render_tab_modules_output(array $options): void
	{
		$is_pro = Capabilities::is_pro($options);

		echo '<h2 style="margin-top:16px;">' . esc_html__('模块与输出', 'content-ready') . '</h2>';
		self::render_badge_legend();

		self::open_form('modules_output');

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row">前端渲染（可视化模块）</th><td>';
		$inject = !empty($options['frontend']['inject_enabled']);
		echo '<label><input type="checkbox" name="frontend_inject_enabled" value="1"' . checked($inject, true, false) . ' /> 启用前端渲染（基于 the_content）</label>';
		echo '<p class="description">开启后，插件会将已生成模块渲染到文章/页面前端正文区域（主题需输出 the_content）。关闭后仅保留编辑器内展示与检查；结构化输出可独立开关。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">前端样式</th><td>';
		$css = !empty($options['frontend']['css_enabled']);
		echo '<label><input type="checkbox" name="frontend_css_enabled" value="1"' . checked($css, true, false) . ' /> 启用轻量预设样式</label>';
		echo '<p class="description">提供基础排版（标题/列表/表格等）。如你的主题已有完整样式，可关闭以避免冲突。</p>';
		echo '</td></tr>';

		echo '<tr><th scope="row">结构化输出（JSON-LD）</th><td>';
		$so_enabled = !empty($options['structured_output']['enabled']);
		echo '<label><input type="checkbox" name="structured_output_enabled" value="1"' . checked($so_enabled, true, false) . ' /> 启用</label>';

		$so_mode = sanitize_key((string)($options['structured_output']['mode'] ?? 'auto'));
		if (!in_array($so_mode, ['auto', 'force', 'off'], true)) {
			$so_mode = 'auto';
		}

		echo '<div style="margin-top:8px;">';
		echo '<span style="display:inline-block;min-width:72px;color:#111827;font-weight:600;">模式</span> ';
		if ($is_pro) {
			echo '<select name="structured_output_mode">';
			foreach (['auto' => '智能兼容（默认）', 'force' => '强制输出', 'off' => '关闭'] as $k => $label) {
				echo '<option value="' . esc_attr($k) . '"' . selected($so_mode, $k, false) . '>' . esc_html($label) . '</option>';
			}
			echo '</select>';
		} else {
			echo '<select disabled>';
			foreach (['auto' => '智能兼容（默认）', 'force' => '强制输出', 'off' => '关闭'] as $k => $label) {
				echo '<option value="' . esc_attr($k) . '"' . selected($so_mode, $k, false) . '>' . esc_html($label) . '</option>';
			}
			echo '</select>';
			echo '<span style="color:#6b7280;margin-left:8px;">Free：固定为智能兼容；Pro 可选强制输出/关闭。</span>';
		}
		echo '</div>';

		echo '<p class="description">用于输出 Article/FAQ 等结构化（JSON-LD）。智能兼容会在检测到站点已有其他结构化输出时自动暂停本插件输出，以避免重复；检测范围仅包含页面头部与底部（wp_head + wp_footer）的 application/ld+json。</p>';
		echo '<p class="description">提示：首次发布后会自动做一次兼容性检查；检查完成前，智能兼容可能会短暂不输出结构化（通常数十秒）。</p>';
		echo '</td></tr>';

		echo '</tbody></table>';

		ModulesSection::render($options);

		self::close_form();
	}

	private static function render_tab_page_types(array $options): void
	{
		echo '<h2 style="margin-top:16px;">' . esc_html__('页面类型规则', 'content-ready') . '</h2>';
		self::open_form('page_types');
		PageTypesSection::render($options);
		self::close_form();
	}

	private static function render_tab_license(array $options): void
	{
		echo '<h2 style="margin-top:16px;">' . esc_html__('许可证', 'content-ready') . ' <span class="cr-badge-wrap">' . Badges::pro() . '</span></h2>';
		self::open_form('license');

		$key = (string)($options['license']['key'] ?? '');
		$status = sanitize_key((string)($options['license']['status'] ?? 'unlicensed'));
		$plan = sanitize_key((string)($options['license']['plan'] ?? 'free'));
		$base_domain = sanitize_text_field((string)($options['license']['base_domain'] ?? ''));
		$checked_at = (int)($options['license']['checked_at'] ?? 0);
		$expires_at = (int)($options['license']['expires_at'] ?? 0);
		$grace_until = (int)($options['license']['grace_until'] ?? 0);
		$last_error = sanitize_text_field((string)($options['license']['last_error'] ?? ''));
		$status_label = [
			'unlicensed' => '未填写',
			'unverified' => '待校验',
			'active' => '已激活',
			'grace' => '宽限期（临时可用）',
			'disabled' => '已禁用',
			'invalid_key' => 'Key 无效',
			'invalid_domain' => '域名不匹配',
			'error' => '校验失败（网络/服务）',
		][$status] ?? $status;

		echo '<p class="description">v1.1：填写 key 后必须通过在线校验才会解锁 Pro；Key 绑定“主域名”，允许无限子域；网络故障可按服务端下发宽限期临时继续使用（但被禁用会立即降级）。</p>';
		echo '<table class="form-table" role="presentation"><tbody>';
		echo '<tr><th scope="row"><label for="cr_license_key">License Key（许可证密钥）</label></th><td>';
		echo '<input class="regular-text" id="cr_license_key" name="license_key" value="' . esc_attr($key) . '" />';
		echo '<p class="description">状态：' . esc_html($status_label) . '；套餐：' . esc_html($plan !== '' ? $plan : 'free') . '</p>';
		if ($base_domain !== '') {
			echo '<p class="description">绑定域名：' . esc_html($base_domain) . '（允许其所有子域）</p>';
		}
		if ($last_error !== '') {
			echo '<p class="description" style="color:#b91c1c;">最近错误：' . esc_html($last_error) . '</p>';
		}
		echo '</td></tr>';
		echo '</tbody></table>';

		self::close_form();

		// 手动触发“立即校验”（不修改 key）。
		echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
		wp_nonce_field('cr_verify_license');
		echo '<input type="hidden" name="action" value="cr_verify_license" />';
		submit_button('立即校验', 'secondary', 'submit', false);
		echo '</form>';

		$cached = Updater::get_cached_update_info();
		$cached_version = sanitize_text_field((string)($cached['version'] ?? ''));
		$cached_details_url = isset($cached['details_url']) ? esc_url_raw((string)$cached['details_url']) : '';
		$cached_checked_at = (int)($cached['checked_at'] ?? 0);
		$current_version = defined('CR_VERSION') ? (string)CR_VERSION : '';

		echo '<div style="margin-top:16px;padding:12px;border:1px solid #e5e7eb;background:#fff;">';
		echo '<p style="margin:0 0 8px;font-weight:600;">更新</p>';
		echo '<p class="description" style="margin:0 0 8px;">';
		echo '当前版本：' . esc_html($current_version !== '' ? $current_version : 'unknown') . '；';
		echo '最新版本：' . esc_html($cached_version !== '' ? $cached_version : '未检查') . '；';
		echo '上次检查：' . esc_html($cached_checked_at > 0 ? wp_date('Y-m-d H:i', $cached_checked_at) : '无');
		echo '</p>';

		echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
		wp_nonce_field('cr_check_updates');
		echo '<input type="hidden" name="action" value="cr_check_updates" />';
		submit_button('检查更新', 'secondary', 'submit', false);
		echo ' <a class="button" href="' . esc_url(admin_url('update-core.php')) . '">打开 WordPress 更新</a>';
		$download_page_url = $cached_details_url !== '' ? $cached_details_url : 'https://cr-license.imsxx.com/v1/update/details';
		echo ' <a class="button" href="' . esc_url($download_page_url) . '" target="_blank" rel="noopener noreferrer">查看更新日志（下载）</a>';
		echo '</form>';

		echo '<p class="description" style="margin:8px 0 0;">提示：检查更新只刷新状态，不会自动安装；下载请打开更新日志（含下载链接）。</p>';
		echo '</div>';

		PlanCompare::render($options);
	}

	private static function render_tab_import_export(): void
	{
		echo '<h2 style="margin-top:16px;">' . esc_html__('导入导出', 'content-ready') . '</h2>';
		ImportExportSection::render();
	}

	private static function render_tab_advanced(array $options): void
	{
		echo '<h2 style="margin-top:16px;">' . esc_html__('高级/维护', 'content-ready') . '</h2>';
		self::open_form('advanced');

		echo '<table class="form-table" role="presentation"><tbody>';

		echo '<tr><th scope="row">卸载清理</th><td>';
		$cleanup = !empty($options['uninstall']['cleanup']);
		echo '<label><input type="checkbox" name="uninstall_cleanup" value="1"' . checked($cleanup, true, false) . ' /> 卸载插件时清理数据（站点设置 + 已生成内容数据）</label>';
		echo '<p class="description">默认关闭；开启后卸载会删除站点配置与文章的插件数据（用于存储已生成模块）。</p>';
		echo '</td></tr>';

		echo '</tbody></table>';

		self::close_form();
	}
}
