<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Modules\Config as ModulesConfig;
use ContentReady\Modules\Dispatch as ModulesDispatch;
use ContentReady\PageTypes\Config as PageTypesConfig;
use ContentReady\Licensing\Capabilities;

final class Settings
{
	public const OPTION_KEY = 'cr_options_v1';
	public const CONFIG_VERSION = 1;

	public static function register(): void
	{
		add_action('admin_menu', [self::class, 'add_menu']);
		add_action('admin_enqueue_scripts', [self::class, 'enqueue_assets']);
		add_action('admin_post_cr_save_settings', [self::class, 'handle_save']);
		ImportExport::register();
	}

	public static function enqueue_assets(string $hook): void
	{
		$is_settings = strpos($hook, 'content-ready-settings') !== false;
		$is_assessment = strpos($hook, 'content-ready-assessment') !== false;
		$is_automation = strpos($hook, 'content-ready-automation') !== false;
		if (!$is_settings && !$is_assessment && !$is_automation) {
			return;
		}

		wp_enqueue_style(
			'content-ready-badges',
			CR_PLUGIN_URL . 'assets/badges.css',
			[],
			(string)(filemtime(CR_PLUGIN_DIR . 'assets/badges.css') ?: CR_VERSION)
		);

		if ($is_settings) {
			wp_enqueue_media();

			wp_enqueue_script(
				'content-ready-admin-settings',
				CR_PLUGIN_URL . 'assets/admin-settings.js',
				[],
				(string)(filemtime(CR_PLUGIN_DIR . 'assets/admin-settings.js') ?: CR_VERSION),
				true
			);

			wp_enqueue_style(
				'content-ready-admin-modules-cards',
				CR_PLUGIN_URL . 'assets/admin-modules-cards.css',
				[],
				(string)(filemtime(CR_PLUGIN_DIR . 'assets/admin-modules-cards.css') ?: CR_VERSION)
			);
		}
	}

	public static function maybe_seed_defaults(): void
	{
		if (get_option(self::OPTION_KEY, null) !== null) {
			return;
		}

		add_option(self::OPTION_KEY, self::default_options(), '', false);
	}

	public static function get_options(): array
	{
		$options = get_option(self::OPTION_KEY, []);
		$options = is_array($options) ? $options : [];
		$options = Capabilities::normalize_options($options);

		// 固定模块顺序与位置（不让站长用户配置排序/位置，避免心智负担）。
		$options['modules_order'] = ModulesConfig::default_order();
		$options['modules'] = ModulesConfig::normalize_modules_config($options['modules'] ?? null);

		// 模块调度策略（Pro-only）：用于“启用很多模块”与“固定轻量化组合”两类用户。
		$is_pro = Capabilities::is_pro($options);
		$gen = is_array($options['generation'] ?? null) ? $options['generation'] : [];
		$raw_mode = isset($gen['dispatch_mode']) ? (string)$gen['dispatch_mode'] : '';
		$mode = ModulesDispatch::normalize_mode($raw_mode, $is_pro);

		// 兼容旧版本：只有 adaptive_enabled 时，自动映射为 adaptive。
		if ($raw_mode === '' && !array_key_exists('dispatch_mode', $gen) && !empty($gen['adaptive_enabled'])) {
			$mode = ModulesDispatch::MODE_ADAPTIVE;
		}

		if (!is_array($options['generation'] ?? null)) {
			$options['generation'] = [];
		}
		$options['generation']['dispatch_mode'] = $mode;
		$options['generation']['adaptive_enabled'] = ($mode === ModulesDispatch::MODE_ADAPTIVE);

		if ($mode === ModulesDispatch::MODE_LIGHTWEIGHT) {
			$options['modules'] = ModulesDispatch::apply_lightweight_modules_config($options['modules']);
		}

		return $options;
	}

	public static function get_raw_options(): array
	{
		$options = get_option(self::OPTION_KEY, []);
		return is_array($options) ? $options : [];
	}

	public static function update_options(array $options): void
	{
		update_option(self::OPTION_KEY, $options, false);
	}

	public static function default_options(): array
	{
		return [
			'config_version' => self::CONFIG_VERSION,
			'checks' => [
				'auto_on_publish' => true,
			],
			'ui' => [
				'editor_entry_label' => '内容就绪',
			],
			'site' => [
				'display_name' => '',
				'logo_url' => '',
				'locale' => '',
				'date_format' => 'wp', // wp | PHP date() format
				'author_mode' => 'wp', // wp | brand
				'brand_author_name' => '',
			],
			'license' => [
				'key' => '',
				'status' => 'unlicensed', // unlicensed | unverified | active | grace | disabled | invalid_key | invalid_domain | error
				'plan' => 'free', // free | pro | max
				'base_domain' => '',
				'checked_at' => 0,
				'expires_at' => 0,
				'ttl_sec' => 0,
				'grace_sec' => 0,
				'grace_until' => 0,
				'last_ok_at' => 0,
				'last_ok_plan' => '',
				'last_http_status' => 0,
				'last_error' => '',
				'last_message' => '',
			],
			'ai' => [
				'base_url' => '',
				'api_key' => '',
				'model' => '',
				'mode' => 'standard',
			],
			'generation' => [
				'adaptive_enabled' => false, // Pro：自适应模块调用（解决模块过多导致结构膨胀）
				'dispatch_mode' => 'off', // off | adaptive | lightweight（Pro）
				'auto_geo_on_publish' => false, // Pro：发布时自动 GEO（异步）
				'backfill_enabled' => false, // Pro：老文章回填
				'backfill_mode' => 'once', // once | daily
				'backfill_daily_limit' => 10,
				'backfill_hour' => 3, // 建议执行时段（0-23）
			],
			'privacy' => [
				'send_scope' => 'full', // full | title_summary | selected_modules
			],
			'uninstall' => [
				'cleanup' => false,
			],
			'frontend' => [
				'inject_enabled' => true,
				'css_enabled' => true,
			],
			'structured_output' => [
				'enabled' => true,
				'mode' => 'auto', // auto | force | off
			],
			'modules_order' => ModulesConfig::default_order(),
			'modules' => ModulesConfig::default_modules_config(),
			'page_types' => PageTypesConfig::default_page_types(),
		];
	}

	public static function add_menu(): void
	{
		// 父菜单：指向 Dashboard
		add_menu_page(
			'Content Ready',
			'Content Ready',
			'manage_options',
			'content-ready',
			[Dashboard::class, 'render_page'],
			'dashicons-yes',
			58
		);

		// 子菜单：看板（与父菜单同 slug，作为默认首页）
		add_submenu_page(
			'content-ready',
			'看板',
			'看板',
			'manage_options',
			'content-ready',
			[Dashboard::class, 'render_page']
		);

		// 子菜单：自动化（放在看板下，开始评估上）
		add_submenu_page(
			'content-ready',
			'自动化',
			'自动化',
			'manage_options',
			'content-ready-automation',
			[AutomationPage::class, 'render_page']
		);

		// 子菜单：开始评估
		add_submenu_page(
			'content-ready',
			'开始评估',
			'开始评估',
			'manage_options',
			'content-ready-assessment',
			[AssessmentPage::class, 'render_page']
		);

		// 子菜单：设置
		add_submenu_page(
			'content-ready',
			'设置',
			'设置',
			'manage_options',
			'content-ready-settings',
			[self::class, 'render_page']
		);
	}

	public static function render_page(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have permission to access this page.', 'content-ready'));
		}

		$notice = isset($_GET['cr_notice']) ? sanitize_text_field((string)$_GET['cr_notice']) : '';
		$tab = isset($_GET['tab']) ? sanitize_key((string)$_GET['tab']) : 'basic';
		$options = self::get_options();
		SettingsPage::render($notice, $options, $tab);
	}

	public static function handle_save(): void
	{
		SettingsSaver::handle_save();
	}
}
