<?php

declare(strict_types=1);

namespace ContentReady\Admin;

final class DeactivationFeedback
{
	private const INSTALL_META_OPTION = 'cr_install_meta_v1';
	private const FEEDBACK_OPTION = 'cr_deactivation_feedback_v1';
	private const USED_POSTS_COUNT_TRANSIENT = 'cr_used_posts_count_v1';

	public static function register(): void
	{
		add_action('admin_enqueue_scripts', [self::class, 'enqueue_assets']);
		add_action('wp_ajax_cr_submit_deactivation_feedback', [self::class, 'handle_submit']);
	}

	public static function on_activate(): void
	{
		self::maybe_seed_install_meta();
	}

	private static function maybe_seed_install_meta(): void
	{
		$meta = get_option(self::INSTALL_META_OPTION, null);
		$meta = is_array($meta) ? $meta : [];

		if (!isset($meta['installed_at']) || !is_int($meta['installed_at']) || $meta['installed_at'] <= 0) {
			$meta['installed_at'] = (int)current_time('timestamp');
		}
		if (!isset($meta['first_version']) || !is_string($meta['first_version']) || $meta['first_version'] === '') {
			$meta['first_version'] = defined('CR_VERSION') ? (string)CR_VERSION : '';
		}

		update_option(self::INSTALL_META_OPTION, $meta, false);
	}

	public static function enqueue_assets(string $hook): void
	{
		if ($hook !== 'plugins.php') {
			return;
		}
		if (!current_user_can('activate_plugins')) {
			return;
		}

		self::maybe_seed_install_meta();

		wp_enqueue_style(
			'content-ready-admin-deactivation-feedback',
			CR_PLUGIN_URL . 'assets/admin-deactivation-feedback.css',
			[],
			CR_VERSION
		);

		wp_enqueue_script(
			'content-ready-admin-deactivation-feedback',
			CR_PLUGIN_URL . 'assets/admin-deactivation-feedback.js',
			[],
			CR_VERSION,
			true
		);

		$payload = [
			'ajaxUrl' => admin_url('admin-ajax.php'),
			'nonce' => wp_create_nonce('cr_deactivation_feedback_v1'),
			'pluginBasename' => plugin_basename(CR_PLUGIN_FILE),
			'pluginVersion' => defined('CR_VERSION') ? (string)CR_VERSION : '',
			'wpVersion' => get_bloginfo('version'),
			'installedDays' => self::get_installed_days(),
			'usedPostsCount' => self::get_used_posts_count(),
		];

		wp_add_inline_script(
			'content-ready-admin-deactivation-feedback',
			'window.ContentReadyDeactivationFeedback = ' . wp_json_encode($payload) . ';',
			'before'
		);
	}

	private static function get_installed_days(): int
	{
		$meta = get_option(self::INSTALL_META_OPTION, null);
		$meta = is_array($meta) ? $meta : [];
		$installed_at = isset($meta['installed_at']) && is_int($meta['installed_at']) ? $meta['installed_at'] : 0;
		if ($installed_at <= 0) {
			return -1;
		}

		$now = (int)current_time('timestamp');
		$days = (int)floor(max(0, $now - $installed_at) / DAY_IN_SECONDS);
		return $days;
	}

	private static function get_used_posts_count(): int
	{
		$cached = get_transient(self::USED_POSTS_COUNT_TRANSIENT);
		if (is_int($cached) && $cached >= 0) {
			return $cached;
		}

		global $wpdb;
		$postmeta = $wpdb->postmeta;
		$posts = $wpdb->posts;

		// 统计已有 cr_meta_v1 的文章数量，用于停用/卸载前提醒用户影响范围（做短期缓存，避免每次打开插件列表都跑一遍）。
		$sql = "
			SELECT COUNT(DISTINCT pm.post_id)
			FROM {$postmeta} pm
			INNER JOIN {$posts} p ON p.ID = pm.post_id
			WHERE pm.meta_key = %s
			  AND p.post_type <> %s
			  AND p.post_status NOT IN ('trash', 'auto-draft')
		";

		$count = (int)$wpdb->get_var($wpdb->prepare($sql, 'cr_meta_v1', 'revision'));
		$count = max(0, $count);

		set_transient(self::USED_POSTS_COUNT_TRANSIENT, $count, 6 * HOUR_IN_SECONDS);
		return $count;
	}

	public static function handle_submit(): void
	{
		if (!current_user_can('activate_plugins')) {
			wp_send_json_error(['message' => 'forbidden'], 403);
		}

		check_ajax_referer('cr_deactivation_feedback_v1', 'nonce');

		$action_type = isset($_POST['action_type']) ? sanitize_key((string)$_POST['action_type']) : '';
		if (!in_array($action_type, ['deactivate', 'delete'], true)) {
			wp_send_json_error(['message' => 'invalid_action_type'], 400);
		}

		$reason = isset($_POST['reason']) ? sanitize_key((string)$_POST['reason']) : '';
		$allowed_reasons = [
			'not_working',
			'hard_to_use',
			'missing_features',
			'temporary',
			'conflict',
			'other',
		];
		if (!in_array($reason, $allowed_reasons, true)) {
			$reason = 'other';
		}

		$details = isset($_POST['details']) ? sanitize_textarea_field((string)$_POST['details']) : '';

		$item = [
			'ts' => (int)current_time('timestamp'),
			'action_type' => $action_type,
			'reason' => $reason,
			'details' => $details,
			'installed_days' => self::get_installed_days(),
			'used_posts_count' => self::get_used_posts_count(),
			'plugin_version' => defined('CR_VERSION') ? (string)CR_VERSION : '',
			'wp_version' => get_bloginfo('version'),
			'php_version' => PHP_VERSION,
		];

		$list = get_option(self::FEEDBACK_OPTION, []);
		$list = is_array($list) ? $list : [];
		array_unshift($list, $item);
		$list = array_slice($list, 0, 50);

		update_option(self::FEEDBACK_OPTION, $list, false);

		wp_send_json_success(['ok' => true]);
	}
}
