<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Licensing\DailyQuota;

/**
 * Dashboard 页面渲染。
 */
final class DashboardPage
{
	/**
	 * 渲染 Dashboard 页面。
	 *
	 * @param array $stats Dashboard::get_stats() 返回的统计数据
	 */
	public static function render(array $stats): void
	{
		$total       = (int)($stats['total_posts'] ?? 0);
		$generated   = (int)($stats['geo_generated'] ?? 0);
		$missing     = (int)($stats['geo_missing'] ?? 0);
		$edited      = (int)($stats['geo_edited'] ?? 0);
		$ai_configured = !empty($stats['ai_configured']);
		$ai_model    = (string)($stats['ai_model'] ?? '');
		$adaptive_enabled = !empty($stats['adaptive_enabled']);
		$dispatch_mode = sanitize_key((string)($stats['dispatch_mode'] ?? ($adaptive_enabled ? 'adaptive' : 'off')));
		if (!in_array($dispatch_mode, ['off', 'adaptive', 'lightweight'], true)) {
			$dispatch_mode = $adaptive_enabled ? 'adaptive' : 'off';
		}
		$structured_enabled = !empty($stats['structured_enabled']);
		$structured_mode = (string)($stats['structured_mode'] ?? 'auto');
		$quota_plan = (string)($stats['quota_plan'] ?? 'free');
		$quota_used = (int)($stats['quota_used'] ?? 0);
		$quota_limit = (int)($stats['quota_limit'] ?? 0);
		$enabled     = (int)($stats['modules_enabled'] ?? 0);
		$disabled    = (int)($stats['modules_disabled'] ?? 0);
		$recent      = is_array($stats['recent_generated'] ?? null) ? $stats['recent_generated'] : [];
		$issues      = is_array($stats['issues'] ?? null) ? $stats['issues'] : [];

		wp_enqueue_style(
			'content-ready-badges',
			CR_PLUGIN_URL . 'assets/badges.css',
			[],
			CR_VERSION
		);

		wp_enqueue_style(
			'content-ready-admin-dashboard',
			CR_PLUGIN_URL . 'assets/admin-dashboard.css',
			[],
			CR_VERSION
		);

		echo '<div class="wrap cr-dashboard">';
		echo '<h1>' . esc_html__('Content Ready｜看板', 'content-ready') . '</h1>';

		// Free：常驻 Pro 提示卡片（避免“关掉后找不到入口”的问题）
		if ($quota_plan === 'free') {
			$license_url = admin_url('admin.php?page=content-ready-settings&tab=license');
			$compare_url = admin_url('admin.php?page=content-ready-settings&tab=license');
			$automation_url = admin_url('admin.php?page=content-ready-automation');

			echo '<div class="cr-dashboard-section">';
			echo '<h2>升级到 Pro（可选）</h2>';
			echo '<p class="description">当前为免费版。Pro 可解锁：自动化（发布后自动 GEO / 老文章批量回填）、更多内容模块、模块调度策略、结构化输出模式（强制输出/关闭）以及更高的每日配额。</p>';
			echo '<p>';
			echo '<a class="button button-primary" href="' . esc_url($license_url) . '">我已购买，去填写 License Key</a> ';
			echo '<a class="button" href="' . esc_url($compare_url) . '">查看免费版 vs Pro 对比</a> ';
			echo '<a class="button" href="' . esc_url($automation_url) . '">查看自动化（Pro）</a>';
			echo '</p>';
			echo '</div>';
		}

		// 问题提示
		if (!empty($issues)) {
			echo '<div class="cr-dashboard-issues">';
			foreach ($issues as $issue) {
				echo '<div class="notice notice-warning inline"><p>' . esc_html($issue) . '</p></div>';
			}
			echo '</div>';
		}

		// 统计卡片
		echo '<div class="cr-dashboard-cards">';

		self::render_card('📄', '总文章数', (string)$total, '已发布的文章和页面');
		self::render_card('✅', '已生成 GEO', (string)$generated, '以“可见更新时间 / 摘要”等稳定模块作为生成完成标记', 'success');
		self::render_card('⚠️', '待处理', (string)$missing, '未生成或有模块缺失', $missing > 0 ? 'warning' : '');

		echo '</div>';

		// 模块状态
		echo '<div class="cr-dashboard-cards">';
		self::render_card('🧩', '启用模块', (string)$enabled, '当前启用的 GEO 模块');
		self::render_card('🚫', '禁用模块', (string)$disabled, '当前禁用的 GEO 模块', $disabled > 0 ? 'muted' : '');
		self::render_card('✏️', '已修改', (string)$edited, '生成后被手动编辑过的文章', $edited > 0 ? 'warning' : '');
		echo '</div>';

		// 配置概览
		$ai_value = $ai_configured ? '已配置' : '未配置';
		$ai_desc = $ai_configured ? ('模型：' . $ai_model) : '请在设置页填写接口地址/Key/模型';

		$dispatch_value = [
			'off' => '关闭',
			'adaptive' => '自适应',
			'lightweight' => '轻量化',
		][$dispatch_mode] ?? '关闭';
		$dispatch_desc = [
			'off' => '完全按你启用的模块生成',
			'adaptive' => '根据正文信号自动跳过不适合的模块',
			'lightweight' => '固定推荐组合减少 AI 调用次数（更省 Token）',
		][$dispatch_mode] ?? '';

		$so_mode_label = [
			'auto' => '智能兼容',
			'force' => '强制输出',
			'off' => '关闭',
		][$structured_mode] ?? $structured_mode;
		$so_value = $structured_enabled ? $so_mode_label : '关闭';
		$so_desc = $structured_enabled ? ('模式：' . $so_mode_label) : '结构化输出未启用';

		echo '<div class="cr-dashboard-cards">';
		self::render_card('🤖', 'AI 状态', $ai_value, $ai_desc, $ai_configured ? 'success' : 'warning');
		self::render_card('🧠', '模块调度', $dispatch_value, $dispatch_desc, $dispatch_mode === 'off' ? 'muted' : 'success');
		self::render_card('🧾', '结构化输出', $so_value, $so_desc, $structured_enabled ? 'success' : 'muted');

		$quota_value = $quota_limit > 0 ? ($quota_used . ' / ' . $quota_limit) : '不限';
		$quota_desc = $quota_limit > 0
			? ('套餐：' . $quota_plan . '（按“当天首次为某篇文章生成 AI 模块”为 1 篇；同一文章当天多次生成只计 1 篇）')
			: ('套餐：' . $quota_plan . '（0=不限；限额由授权端下发）');
		$quota_variant = ($quota_limit > 0 && $quota_used >= $quota_limit) ? 'warning' : 'muted';
		self::render_card('🧮', '每日配额', $quota_value, $quota_desc, $quota_variant);

		echo '</div>';

		// 每日配额说明（常驻）
		$until_reset = DailyQuota::seconds_until_reset();
		$reset_hint = $until_reset > 60 ? sprintf('约 %d 分钟后重置（按站点时区每日 00:00）。', (int)ceil($until_reset / 60)) : '即将重置（按站点时区每日 00:00）。';
		echo '<div class="cr-dashboard-section">';
		echo '<h2>每日配额说明</h2>';
		echo '<ul style="margin:0;padding-left:18px;color:#374151;">';
		echo '<li>配额来源：在线校验/票据接口会下发当日限额（0 表示不限制）。</li>';
		echo '<li>计数方式：当天首次为某篇文章生成 AI 模块计 1 篇（同篇不重复计数）。</li>';
		echo '<li>授权端默认（当前）：Free=不限（0）、Pro=50/天、Max=200/天（以授权端下发为准，可调整）。</li>';
		echo '<li>重置时间：' . esc_html($reset_hint) . '</li>';
		echo '</ul>';
		echo '</div>';

		// 最近生成
		echo '<div class="cr-dashboard-section">';
		echo '<h2>最近生成</h2>';

		if (empty($recent)) {
			echo '<p class="description">暂无生成记录。</p>';
		} else {
			echo '<table class="widefat striped">';
			echo '<thead><tr><th>文章</th><th>生成时间</th></tr></thead>';
			echo '<tbody>';
			foreach ($recent as $item) {
				$edit_url = get_edit_post_link((int)$item['id']);
				$title = esc_html((string)$item['title']);
				$time = (int)$item['time'];
				$time_str = $time > 0 ? wp_date(get_option('date_format') . ' ' . get_option('time_format'), $time) : '-';

				echo '<tr>';
				echo '<td><a href="' . esc_url((string)$edit_url) . '">' . $title . '</a></td>';
				echo '<td>' . esc_html((string)$time_str) . '</td>';
				echo '</tr>';
			}
			echo '</tbody></table>';
		}

		echo '</div>';

		// 快捷操作
		echo '<div class="cr-dashboard-section">';
		echo '<h2>快捷操作</h2>';
		echo '<p>';
		echo '<a href="' . esc_url(admin_url('admin.php?page=content-ready-settings')) . '" class="button">前往设置</a> ';
		echo '<a href="' . esc_url(admin_url('admin.php?page=content-ready-assessment')) . '" class="button">开始评估</a> ';
		echo '<a href="' . esc_url(admin_url('edit.php')) . '" class="button">管理文章</a>';
		echo '</p>';
		echo '</div>';

		echo '</div>';
	}

	/**
	 * 渲染单个统计卡片。
	 */
	private static function render_card(string $icon, string $label, string $value, string $desc, string $variant = ''): void
	{
		$class = 'cr-dashboard-card';
		if ($variant !== '') {
			$class .= ' cr-dashboard-card--' . $variant;
		}

		echo '<div class="' . esc_attr($class) . '">';
		echo '<div class="cr-dashboard-card__icon">' . $icon . '</div>';
		echo '<div class="cr-dashboard-card__content">';
		echo '<div class="cr-dashboard-card__value">' . esc_html($value) . '</div>';
		echo '<div class="cr-dashboard-card__label">' . esc_html($label) . '</div>';
		echo '<div class="cr-dashboard-card__desc">' . esc_html($desc) . '</div>';
		echo '</div>';
		echo '</div>';
	}
}
