<?php

declare(strict_types=1);

namespace ContentReady\Admin;

use ContentReady\Licensing\Capabilities;
use ContentReady\Licensing\DailyQuota;
use ContentReady\Meta\MetaStore;
use ContentReady\Modules\Registry;

/**
 * Dashboard 数据聚合：统计站点 GEO 内容状态。
 */
final class Dashboard
{
	/**
	 * 获取 Dashboard 统计数据。
	 *
	 * @return array{
	 *     total_posts: int,
	 *     geo_generated: int,
	 *     geo_missing: int,
	 *     geo_edited: int,
	 *     ai_configured: bool,
	 *     ai_model: string,
	 *     adaptive_enabled: bool,
	 *     dispatch_mode: string,
	 *     structured_enabled: bool,
	 *     structured_mode: string,
	 *     quota_plan: string,
	 *     quota_used: int,
	 *     quota_limit: int,
	 *     modules_enabled: int,
	 *     modules_disabled: int,
	 *     recent_generated: array<array{id: int, title: string, time: int}>,
	 *     issues: string[]
	 * }
	 */
	public static function get_stats(): array
	{
		$options = Settings::get_options();
		$modules_config = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$ai_key = (string)($options['ai']['api_key'] ?? '');
		$ai_url = (string)($options['ai']['base_url'] ?? '');
		$ai_model = (string)($options['ai']['model'] ?? '');
		$ai_configured = $ai_key !== '' && $ai_url !== '' && $ai_model !== '';

		$adaptive_enabled = !empty($options['generation']['adaptive_enabled']);
		$dispatch_mode = sanitize_key((string)($options['generation']['dispatch_mode'] ?? 'off'));
		if (!in_array($dispatch_mode, ['off', 'adaptive', 'lightweight'], true)) {
			$dispatch_mode = $adaptive_enabled ? 'adaptive' : 'off';
		}

		$so = is_array($options['structured_output'] ?? null) ? $options['structured_output'] : [];
		$structured_enabled = !empty($so['enabled']);
		$structured_mode = sanitize_key((string)($so['mode'] ?? 'auto'));
		if (!in_array($structured_mode, ['auto', 'force', 'off'], true)) {
			$structured_mode = 'auto';
		}

		// 统计启用/禁用模块数
		$enabled_ids = [];
		$disabled_count = 0;
		foreach (Registry::all_ids() as $module_id) {
			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			if (!empty($cfg['enabled'])) {
				$enabled_ids[] = $module_id;
			} else {
				$disabled_count++;
			}
		}

		// 查询所有已发布的文章/页面
		$post_ids = get_posts([
			'post_type'      => ['post', 'page'],
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'fields'         => 'ids',
		]);
		$total_posts = count($post_ids);

		$geo_generated = 0;
		$geo_missing = 0;
		$geo_edited = 0;
		$recent_list = [];

		foreach ($post_ids as $post_id) {
			$meta = MetaStore::get((int)$post_id);
			$generated_at = (int)($meta['generated_at'] ?? 0);

			$is_complete = self::is_post_geo_complete($meta, $options, $modules_config);
			if ($is_complete) {
				$geo_generated++;
			} else {
				$geo_missing++;
			}

			// 有生成时间时才进入“最近生成”列表
			if ($generated_at > 0) {
				$recent_list[] = [
					'id'    => (int)$post_id,
					'title' => get_the_title((int)$post_id),
					'time'  => $generated_at,
				];
			}

			// 检测是否存在“生成后手动编辑”的内容模块
			foreach (self::get_enabled_content_modules_for_post($meta, $options, $modules_config) as $module_id) {
				$m = is_array($meta['modules'][$module_id] ?? null) ? $meta['modules'][$module_id] : [];
				$gen = (int)($m['generated_at'] ?? 0);
				$edit = (int)($m['edited_at'] ?? 0);
				if ($edit > 0 && $gen > 0 && $edit > $gen) {
					$geo_edited++;
					break;
				}
			}
		}

		// 按时间倒序排列，取最近 5 篇
		usort($recent_list, fn($a, $b) => $b['time'] - $a['time']);
		$recent_generated = array_slice($recent_list, 0, 5);

		// 检测常见问题
		$issues = self::detect_issues($options);

		$quota_plan = Capabilities::plan($options);
		$quota_limit = DailyQuota::limit($options);
		$quota_state = DailyQuota::state();
		$quota_used = (int)($quota_state['used'] ?? 0);

		return [
			'total_posts'      => $total_posts,
			'geo_generated'    => $geo_generated,
			'geo_missing'      => $geo_missing,
			'geo_edited'       => $geo_edited,
			'ai_configured'    => $ai_configured,
			'ai_model'         => $ai_model,
			'adaptive_enabled' => $adaptive_enabled,
			'dispatch_mode'    => $dispatch_mode,
			'structured_enabled' => $structured_enabled,
			'structured_mode'  => $structured_mode,
			'quota_plan'       => $quota_plan,
			'quota_used'       => $quota_used,
			'quota_limit'      => $quota_limit,
			'modules_enabled'  => count($enabled_ids),
			'modules_disabled' => $disabled_count,
			'recent_generated' => $recent_generated,
			'issues'           => $issues,
		];
	}

	/**
	 * 检测常见配置问题。
	 *
	 * @param array $options
	 * @return string[]
	 */
	private static function detect_issues(array $options): array
	{
		$issues = [];

		// AI Key 未配置
		$ai_key = (string)($options['ai']['api_key'] ?? '');
		if ($ai_key === '') {
			$issues[] = 'AI Key 未配置，无法使用 AI 生成功能。';
		}

		// AI 接口地址未配置
		$ai_url = (string)($options['ai']['base_url'] ?? '');
		if ($ai_url === '') {
			$issues[] = 'AI 接口地址未配置。';
		}

		// AI 模型未配置
		$ai_model = (string)($options['ai']['model'] ?? '');
		if ($ai_model === '') {
			$issues[] = 'AI 模型未配置。';
		}

		// 没有启用任何模块
		$modules_config = is_array($options['modules'] ?? null) ? $options['modules'] : [];
		$any_enabled = false;
		foreach (Registry::all_ids() as $module_id) {
			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			if (!empty($cfg['enabled'])) {
				$any_enabled = true;
				break;
			}
		}
		if (!$any_enabled) {
			$issues[] = '没有启用任何模块，请在设置中启用至少一个模块。';
		}

		return $issues;
	}

	/**
	 * 判断某篇文章是否可计入“已生成 GEO”。
	 *
	 * 设计目标：完成度判定稳定、可预测，避免随模块增多而频繁调整。
	 * 判定逻辑：
	 * - 必须存在顶层 generated_at
	 * - 优先使用 last_updated 作为“已生成”标记（稳定/本地生成/always）
	 * - 若 last_updated 不在当前页面类型或被禁用，则回退到 summary（若启用）
	 * - 若两者都不可用，则仅以 generated_at 作为最弱判定
	 */
	private static function is_post_geo_complete(array $meta, array $options, array $modules_config): bool
	{
		$generated_at = (int)($meta['generated_at'] ?? 0);
		if ($generated_at <= 0) {
			return false;
		}

		$checkpoint = self::get_completion_checkpoint_module_id($meta, $options, $modules_config);
		if ($checkpoint === '') {
			return true;
		}

		$data = $meta['modules'][$checkpoint]['data'] ?? [];
		return self::is_module_data_present($checkpoint, is_array($data) ? $data : []);
	}

	/**
	 * 获取“完成度标记模块”ID（返回空字符串表示仅用 generated_at 判定）。
	 */
	private static function get_completion_checkpoint_module_id(array $meta, array $options, array $modules_config): string
	{
		$enabled = array_flip(self::get_enabled_content_modules_for_post($meta, $options, $modules_config));
		if (isset($enabled['last_updated'])) {
			return 'last_updated';
		}
		if (isset($enabled['summary'])) {
			return 'summary';
		}
		return '';
	}

	/**
	 * 获取某篇文章当前页面类型下启用的内容模块（不含 slug_suggestions）。
	 *
	 * @return string[]
	 */
	private static function get_enabled_content_modules_for_post(array $meta, array $options, array $modules_config): array
	{
		$page_type = is_string($meta['page_type'] ?? null) ? (string)$meta['page_type'] : 'article';
		$page_types = is_array($options['page_types'] ?? null) ? $options['page_types'] : [];

		$pt_modules = [];
		foreach ($page_types as $row) {
			if (!is_array($row)) {
				continue;
			}
			if (($row['id'] ?? null) === $page_type && is_array($row['modules'] ?? null)) {
				$pt_modules = array_values($row['modules']);
				break;
			}
		}

		if ($pt_modules === []) {
			$pt_modules = array_values(array_filter(Registry::all_ids(), [Registry::class, 'is_content_module']));
		}

		$out = [];
		foreach ($pt_modules as $module_id) {
			$module_id = sanitize_key((string)$module_id);
			if (!Registry::is_valid_id($module_id) || !Registry::is_content_module($module_id)) {
				continue;
			}
			if ($module_id === 'slug_suggestions') {
				continue;
			}
			$cfg = is_array($modules_config[$module_id] ?? null) ? $modules_config[$module_id] : [];
			if (empty($cfg['enabled'])) {
				continue;
			}
			$out[] = $module_id;
		}

		return $out;
	}

	/**
	 * 判断模块 data 是否“有意义地存在”。
	 */
	private static function is_module_data_present(string $module_id, array $data): bool
	{
		if ($module_id === 'last_updated') {
			return trim((string)($data['date'] ?? '')) !== '';
		}
		if ($module_id === 'summary' || $module_id === 'verdict') {
			return trim((string)($data['text'] ?? '')) !== '';
		}
		return $data !== [];
	}

	/**
	 * 渲染 Dashboard 页面入口。
	 */
	public static function render_page(): void
	{
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have permission to access this page.', 'content-ready'));
		}

		DashboardPage::render(self::get_stats());
	}
}
