(function () {
	function encodeLines(items) {
		if (!items || !items.length) return '';
		return items.join('\n');
	}

	function decodeLines(text) {
		if (!text) return [];
		return text
			.split('\n')
			.map(function (s) {
				return (s || '').trim();
			})
			.filter(Boolean);
	}

	function encodePairs(items, leftKey, rightKey) {
		if (!items || !items.length) return '';
		return items
			.map(function (it) {
				var l = (it && it[leftKey]) || '';
				var r = (it && it[rightKey]) || '';
				return String(l).replace(/\|/g, ' ') + '|' + String(r).replace(/\|/g, ' ');
			})
			.join('\n');
	}

	function decodePairs(text, leftKey, rightKey) {
		var lines = decodeLines(text);
		return lines
			.map(function (line) {
				var idx = line.indexOf('|');
				if (idx === -1) return null;
				var l = line.slice(0, idx).trim();
				var r = line.slice(idx + 1).trim();
				if (!l || !r) return null;
				var out = {};
				out[leftKey] = l;
				out[rightKey] = r;
				return out;
			})
			.filter(Boolean);
	}

	function encodeTable(data) {
		data = data || {};
		var lines = [];
		var caption = (data.caption || '').trim();
		var notes = (data.notes || '').trim();
		if (caption) lines.push('# caption: ' + caption);
		if (notes) lines.push('# notes: ' + notes);

		var cols = Array.isArray(data.columns) ? data.columns : [];
		var rows = Array.isArray(data.rows) ? data.rows : [];
		if (cols && cols.length) {
			lines.push(
				cols
					.map(function (c) {
						return String(c || '').replace(/\|/g, ' ').trim();
					})
					.filter(Boolean)
					.join('|')
			);
		}
		if (rows && rows.length) {
			rows.forEach(function (row) {
				if (!Array.isArray(row)) return;
				lines.push(
					row
						.map(function (c) {
							return String(c || '').replace(/\|/g, ' ').trim();
						})
						.join('|')
				);
			});
		}
		return lines.join('\n');
	}

	function decodeTable(text) {
		var lines = decodeLines(text);
		var caption = '';
		var notes = '';
		var dataLines = [];

		lines.forEach(function (line) {
			if (!line) return;
			if (line.indexOf('#') === 0) {
				var s = line.slice(1).trim();
				if (s.toLowerCase().indexOf('caption:') === 0) caption = s.slice(8).trim();
				else if (s.toLowerCase().indexOf('notes:') === 0) notes = s.slice(6).trim();
				return;
			}
			dataLines.push(line);
		});

		var columns = [];
		var rows = [];
		if (dataLines.length) {
			columns = dataLines[0]
				.split('|')
				.map(function (s) {
					return (s || '').trim();
				})
				.filter(Boolean);

			dataLines.slice(1).forEach(function (line) {
				var row = line
					.split('|')
					.map(function (s) {
						return (s || '').trim();
					});
				if (!row.length) return;
				if (columns.length) {
					if (row.length < columns.length) {
						while (row.length < columns.length) row.push('');
					} else if (row.length > columns.length) {
						row = row.slice(0, columns.length);
					}
				}
				rows.push(row);
			});
		}

		return { caption: caption, columns: columns, rows: rows, notes: notes };
	}

	function encodeDraft(moduleId, data) {
		data = data || {};
		if (moduleId === 'summary') return data.text || '';
		if (moduleId === 'verdict') return data.text || '';
		if (moduleId === 'tldr' || moduleId === 'limitations') return encodeLines(data.items || []);
		if (moduleId === 'faq') return encodePairs(data.items || [], 'q', 'a');
		if (moduleId === 'references') return encodePairs(data.items || [], 'title', 'url');
		if (moduleId === 'last_updated') return data.date || '';
		if (moduleId === 'table') return encodeTable(data);
		if (moduleId === 'slug_suggestions') return encodeLines(data.suggestions || []);
		return '';
	}

	function decodeDraft(moduleId, text) {
		if (moduleId === 'summary') return { text: text || '' };
		if (moduleId === 'verdict') return { text: text || '' };
		if (moduleId === 'tldr' || moduleId === 'limitations') return { items: decodeLines(text) };
		if (moduleId === 'faq') return { items: decodePairs(text, 'q', 'a') };
		if (moduleId === 'references') return { items: decodePairs(text, 'title', 'url') };
		if (moduleId === 'last_updated') return { date: text || '' };
		if (moduleId === 'table') return decodeTable(text);
		if (moduleId === 'slug_suggestions') return { suggestions: decodeLines(text) };
		return {};
	}

	function moduleStatus(moduleMeta) {
		if (!moduleMeta) return '缺失';
		var gen = moduleMeta.generated_at || 0;
		var edit = moduleMeta.edited_at || 0;
		var data = moduleMeta.data || {};
		var hasData = data && Object.keys(data).length > 0;
		if (!gen || !hasData) return '缺失';
		if (edit > gen) return '已修改';
		return '已生成';
	}

	function flash(el) {
		if (!el) return;
		var prevBg = el.style.backgroundColor;
		var prevShadow = el.style.boxShadow;
		el.style.backgroundColor = '#fffbeb';
		el.style.boxShadow = '0 0 0 2px #f59e0b inset';
		window.setTimeout(function () {
			el.style.backgroundColor = prevBg;
			el.style.boxShadow = prevShadow;
		}, 1200);
	}

	function scrollToModule(moduleId) {
		if (!moduleId) return false;
		var el = document.getElementById('cr-classic-module-' + moduleId);
		if (!el) return false;
		try {
			el.scrollIntoView({ behavior: 'smooth', block: 'start' });
		} catch (e) {
			el.scrollIntoView();
		}
		flash(el);
		var ta = el.querySelector('textarea');
		if (ta && ta.focus) {
			window.setTimeout(function () {
				ta.focus();
			}, 50);
		}
		return true;
	}

	window.CR_CLASSIC_UTILS = {
		encodeDraft: encodeDraft,
		decodeDraft: decodeDraft,
		moduleStatus: moduleStatus,
		encodeTable: encodeTable,
		decodeTable: decodeTable,
		scrollToModule: scrollToModule,
	};
})();
